"use client"

import type React from "react"

import Link from "next/link"
import PageHeader from "@/components/page-header"
import { useState } from "react"
import { useRouter } from "next/navigation"
import { SafeImage } from "@/components/safe-image"

// Image par défaut à utiliser si l'image de l'article n'existe pas
const DEFAULT_IMAGE = "/modern-apartment-living.png"

// Liste des articles avec vérification des images
const articles = [
  {
    slug: "guide-proprietaire-bailleur-2024",
    title: "Guide Complet du Propriétaire Bailleur 2025",
    description: "Tout ce que vous devez savoir pour gérer efficacement votre bien immobilier en location en 2025.",
    date: "15 mars 2025",
    category: "Guide",
    image: "/landlord-guide-concept.png",
  },
  {
    slug: "obligations-energetiques-proprietaires-2024",
    title: "Nouvelles Obligations Énergétiques pour les Propriétaires Bailleurs en 2024",
    description:
      "Découvrez les nouvelles réglementations énergétiques et comment vous y conformer pour éviter les sanctions.",
    date: "22 février 2024",
    category: "Réglementation",
    image: "/energy-efficiency-report.png",
  },
  {
    slug: "investir-immobilier-ile-de-france-2024",
    title: "Où Investir dans l'Immobilier en Île-de-France en 2024 ?",
    description: "Analyse des meilleures opportunités d'investissement immobilier en Île-de-France pour 2024.",
    date: "5 mars 2024",
    category: "Investissement",
    image: "/paris-apartment-view.png",
  },
  {
    slug: "assurance-loyers-impayes-2024",
    title: "Comment Choisir la Meilleure Assurance Loyers Impayés en 2024",
    description: "Guide complet pour comprendre et sélectionner l'assurance loyers impayés adaptée à votre situation.",
    date: "18 mars 2024",
    category: "Assurance",
    image: "/keys-and-contract.png",
  },
  {
    slug: "optimisation-fiscale-proprietaires-2024",
    title: "Optimisation Fiscale pour Propriétaires Bailleurs : Les Stratégies 2024",
    description:
      "Découvrez comment optimiser votre fiscalité immobilière et maximiser la rentabilité de vos investissements.",
    date: "2 avril 2024",
    category: "Fiscalité",
    image: "/property-tax-documents.png",
  },
  {
    slug: "travaux-renovation-valeur-locative",
    title: "Quels Travaux de Rénovation Privilégier pour Augmenter la Valeur Locative ?",
    description: "Guide des rénovations les plus rentables pour maximiser le rendement de votre bien immobilier.",
    date: "15 avril 2024",
    category: "Rénovation",
    image: "/apartment-renovation-in-progress.png",
  },
  {
    slug: "colocation-avantages-fiscaux-gestion",
    title: "Colocation : Avantages Fiscaux et Gestion Optimale en 2024",
    description:
      "Découvrez comment maximiser votre rentabilité locative grâce à la colocation et optimiser sa gestion.",
    date: "28 avril 2024",
    category: "Gestion locative",
    image: "/colocation-moderne-salon.png",
  },
  {
    slug: "lmnp-lmp-comparaison-statuts",
    title: "LMNP vs LMP : Quel Statut Choisir pour Votre Investissement Locatif en 2024 ?",
    description:
      "Analyse comparative des statuts LMNP et LMP, leurs avantages fiscaux et critères de choix pour optimiser votre investissement immobilier locatif.",
    date: "5 mai 2024",
    category: "Fiscalité",
    image: "/lmnp-lmp-comparison-chart.png",
  },
  {
    slug: "ifi-immobilier-locatif",
    title: "Comment Réduire Votre IFI Grâce à l'Immobilier Locatif en 2024",
    description:
      "Stratégies légales et efficaces pour optimiser votre Impôt sur la Fortune Immobilière grâce à des investissements locatifs judicieux.",
    date: "10 mai 2024",
    category: "Fiscalité",
    image: "/tax-optimization-document.png",
  },
  {
    slug: "dpe-2024-changements-proprietaires",
    title: "DPE 2024 : Ce Qui Change pour les Propriétaires Bailleurs",
    description:
      "Tout savoir sur les nouvelles normes du Diagnostic de Performance Énergétique et leur impact sur la location et la vente de biens immobiliers.",
    date: "15 mai 2024",
    category: "Réglementation",
    image: "/dpe-energy-rating.png",
  },
  {
    slug: "encadrement-loyers-paris-bilan",
    title: "Encadrement des Loyers à Paris : Bilan et Perspectives",
    description:
      "Analyse détaillée du dispositif d'encadrement des loyers à Paris, son efficacité, ses limites et son évolution future.",
    date: "20 mai 2024",
    category: "Réglementation",
    image: "/paris-rental-market.png",
  },
  {
    slug: "obligations-transition-energetique-proprietaires",
    title: "Les Nouvelles Obligations des Propriétaires Face à la Transition Énergétique",
    description:
      "Panorama complet des obligations environnementales pour les propriétaires bailleurs et les mesures à prendre pour s'y conformer.",
    date: "25 mai 2024",
    category: "Réglementation",
    image: "/green-building-transition.png",
  },
  {
    slug: "eviter-impayes-loyer-guide",
    title: "Comment Éviter les Impayés de Loyer : Guide Complet pour Propriétaires",
    description:
      "Stratégies préventives et solutions efficaces pour sécuriser vos revenus locatifs et gérer les situations d'impayés.",
    date: "1 juin 2024",
    category: "Gestion locative",
    image: "/secure-payment-concept.png",
  },
  {
    slug: "erreurs-gestion-locative-eviter",
    title: "Les 10 Erreurs à Éviter dans la Gestion de Votre Bien Locatif",
    description:
      "Découvrez les pièges courants de la gestion locative et comment les éviter pour optimiser votre investissement.",
    date: "5 juin 2024",
    category: "Gestion locative",
    image: "/property-management-mistakes.png",
  },
  {
    slug: "gestion-locative-soi-meme-ou-professionnel",
    title: "Gestion Locative : Faire Soi-même ou Déléguer à un Professionnel ?",
    description:
      "Analyse comparative des avantages et inconvénients de l'autogestion et de la délégation à un gestionnaire professionnel.",
    date: "10 juin 2024",
    category: "Gestion locative",
    image: "/property-management-choice.png",
  },
  {
    slug: "meilleurs-quartiers-paris-investir-2024",
    title: "Les Meilleurs Quartiers de Paris pour Investir en 2024",
    description:
      "Découvrez notre analyse détaillée des quartiers parisiens les plus prometteurs pour un investissement immobilier rentable.",
    date: "15 juin 2024",
    category: "Investissement",
    image: "/paris-neighborhoods-map.png",
  },
  {
    slug: "calculer-rentabilite-investissement-locatif-paris",
    title: "Comment Calculer la Rentabilité d'un Investissement Locatif à Paris",
    description:
      "Guide pratique pour évaluer avec précision la rentabilité de votre projet d'investissement immobilier dans la capitale.",
    date: "20 juin 2024",
    category: "Investissement",
    image: "/investment-calculator-concept.png",
  },
  {
    slug: "investir-ancien-vs-neuf-ile-de-france",
    title: "Investir dans l'Ancien vs le Neuf en Île-de-France",
    description:
      "Analyse comparative des avantages et inconvénients de l'investissement dans l'immobilier ancien et neuf en région parisienne.",
    date: "25 juin 2024",
    category: "Investissement",
    image: "/old-vs-new-property.png",
  },
]

export default function BlogPageClient() {
  const router = useRouter()
  const [searchQuery, setSearchQuery] = useState("")
  const [currentPage, setCurrentPage] = useState(1)
  const articlesPerPage = 9 // 3x3 grid

  // Calculer les articles à afficher sur la page actuelle
  const indexOfLastArticle = currentPage * articlesPerPage
  const indexOfFirstArticle = indexOfLastArticle - articlesPerPage
  const currentArticles = articles.slice(indexOfFirstArticle, indexOfLastArticle)
  const totalPages = Math.ceil(articles.length / articlesPerPage)

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    if (searchQuery.trim()) {
      router.push(`/blog/search?q=${encodeURIComponent(searchQuery.trim())}`)
    }
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <PageHeader title="Blog DOGES" description="Conseils d'experts et actualités en gestion immobilière" />

      <div className="mb-8">
        <form onSubmit={handleSearch} className="relative">
          <input
            type="text"
            placeholder="Rechercher un article..."
            className="w-full p-3 pl-10 border rounded-lg"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
          <svg
            className="absolute left-3 top-3.5 h-5 w-5 text-gray-400"
            fill="none"
            stroke="currentColor"
            viewBox="0 0 24 24"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
            />
          </svg>
          <button
            type="submit"
            className="absolute right-3 top-2 px-4 py-1 bg-blue-600 text-white rounded hover:bg-blue-700"
          >
            Rechercher
          </button>
        </form>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
        {currentArticles.map((article) => (
          <div key={article.slug} className="bg-white rounded-lg shadow-md overflow-hidden">
            <Link href={`/blog/articles/${article.slug}`}>
              <div className="relative h-48">
                <SafeImage
                  src={article.image || DEFAULT_IMAGE}
                  alt={`Image pour ${article.title}`}
                  fill
                  className="object-cover"
                />
              </div>
            </Link>
            <div className="p-6">
              <span className="text-sm text-blue-600 font-semibold">{article.category}</span>
              <Link href={`/blog/articles/${article.slug}`}>
                <h3 className="text-xl font-bold mt-2 mb-3 hover:text-blue-600 transition-colors">{article.title}</h3>
              </Link>
              <p className="text-gray-600 mb-4">{article.description}</p>
              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-500">{article.date}</span>
                <Link
                  href={`/blog/articles/${article.slug}`}
                  className="text-blue-600 hover:underline text-sm font-medium"
                >
                  Lire l'article →
                </Link>
              </div>
            </div>
          </div>
        ))}
      </div>

      <div className="mt-12 flex justify-center">
        <nav className="inline-flex">
          <button
            onClick={() => setCurrentPage((prev) => Math.max(prev - 1, 1))}
            disabled={currentPage === 1}
            className={`px-4 py-2 border border-gray-300 text-sm font-medium ${
              currentPage === 1
                ? "bg-gray-100 text-gray-400 cursor-not-allowed"
                : "bg-white text-gray-700 hover:bg-gray-50"
            }`}
          >
            Précédent
          </button>

          {Array.from({ length: totalPages }, (_, i) => i + 1).map((pageNumber) => (
            <button
              key={pageNumber}
              onClick={() => setCurrentPage(pageNumber)}
              className={`px-4 py-2 border border-gray-300 text-sm font-medium ${
                currentPage === pageNumber
                  ? "bg-blue-600 text-white hover:bg-blue-700"
                  : "bg-white text-gray-700 hover:bg-gray-50"
              }`}
            >
              {pageNumber}
            </button>
          ))}

          <button
            onClick={() => setCurrentPage((prev) => Math.min(prev + 1, totalPages))}
            disabled={currentPage === totalPages}
            className={`px-4 py-2 border border-gray-300 text-sm font-medium ${
              currentPage === totalPages
                ? "bg-gray-100 text-gray-400 cursor-not-allowed"
                : "bg-white text-gray-700 hover:bg-gray-50"
            }`}
          >
            Suivant
          </button>
        </nav>
      </div>
    </div>
  )
}
