import type React from "react"
import type { Metadata } from "next"
import { Inter } from "next/font/google"
import "./globals.css"
import SiteHeader from "@/components/site-header"
import SiteFooter from "@/components/site-footer"
import IntelligentChatbot from "@/components/intelligent-chatbot"
import ScrollToTop from "@/components/scroll-to-top"

const inter = Inter({ subsets: ["latin"] })

export const metadata: Metadata = {
  metadataBase: new URL("https://dogesadb.fr"),
  title: {
    default: "DOGES - Administrateur de biens à Paris | Gestion Locative & Immobilière",
    template: "%s | DOGES",
  },
  description:
    "DOGES, administrateur de biens à Paris depuis plus de 25 ans. Gestion locative, syndic de copropriété, transactions immobilières et assurance loyers impayés.",
  keywords:
    "administrateur de biens Paris, gestion locative Paris, gestion immobilière professionnelle, syndic Paris, assurance loyers impayés",
  authors: [{ name: "DOGES" }],
  creator: "DOGES",
  publisher: "DOGES",
  formatDetection: {
    email: false,
    address: false,
    telephone: false,
  },
  openGraph: {
    type: "website",
    locale: "fr_FR",
    url: "https://dogesadb.fr",
    siteName: "DOGES - Administrateur de biens à Paris",
    title: "DOGES - Administrateur de biens à Paris | Gestion Locative & Immobilière",
    description:
      "DOGES, administrateur de biens à Paris depuis plus de 25 ans. Gestion locative, syndic de copropriété, transactions immobilières et assurance loyers impayés.",
    images: [
      {
        url: "https://dogesadb.fr/og-image.jpg",
        width: 1200,
        height: 630,
        alt: "DOGES - Administrateur de biens à Paris",
      },
    ],
  },
  twitter: {
    card: "summary_large_image",
    title: "DOGES - Administrateur de biens à Paris | Gestion Locative & Immobilière",
    description:
      "DOGES, administrateur de biens à Paris depuis plus de 25 ans. Gestion locative, syndic de copropriété, transactions immobilières et assurance loyers impayés.",
    images: ["https://dogesadb.fr/og-image.jpg"],
    creator: "@doges",
  },
  alternates: {
    canonical: "https://dogesadb.fr",
    languages: {
      "fr-FR": "https://dogesadb.fr",
    },
  },
  verification: {
    google: "verification_token",
  },
    generator: 'ntech'
}

export default function RootLayout({
  children,
}: Readonly<{
  children: React.ReactNode
}>) {
  return (
    <html lang="fr">
      <head>
        <script
          type="application/ld+json"
          dangerouslySetInnerHTML={{
            __html: JSON.stringify({
              "@context": "https://schema.org",
              "@type": "RealEstateAgent",
              name: "DOGES",
              description: "Administrateur de biens à Paris depuis plus de 25 ans",
              url: "https://dogesadb.fr",
              logo: "https://dogesadb.fr/logo.png",
              image: "https://dogesadb.fr/images/office.jpg",
              address: {
                "@type": "PostalAddress",
                streetAddress: "92, Avenue de Saint-Mandé",
                addressLocality: "Paris",
                postalCode: "75012",
                addressCountry: "FR",
              },
              telephone: "+33184809810",
              email: "contact@dogesadb.fr",
              openingHours: ["Mo,Tu,We,Th,Fr 09:00-18:00"],
              areaServed: [
                {
                  "@type": "City",
                  name: "Paris",
                },
                {
                  "@type": "City",
                  name: "Neuilly-sur-Seine",
                },
                {
                  "@type": "City",
                  name: "Levallois-Perret",
                },
              ],
              priceRange: "€€",
              sameAs: [
                "https://www.facebook.com/doges",
                "https://www.linkedin.com/company/doges",
                "https://www.instagram.com/doges",
              ],
              aggregateRating: {
                "@type": "AggregateRating",
                ratingValue: "4.8",
                reviewCount: "127",
              },
              hasOfferCatalog: {
                "@type": "OfferCatalog",
                name: "Services de gestion immobilière",
                itemListElement: [
                  {
                    "@type": "Offer",
                    itemOffered: {
                      "@type": "Service",
                      name: "Gestion Locative",
                      description: "Service complet de gestion locative pour propriétaires",
                      url: "https://dogesadb.fr/services/gestion-locative",
                    },
                  },
                  {
                    "@type": "Offer",
                    itemOffered: {
                      "@type": "Service",
                      name: "Garantie Loyers Impayés",
                      description: "Protection contre les impayés et dégradations",
                      url: "https://dogesadb.fr/services/garantie-loyers-impayes",
                    },
                  },
                  {
                    "@type": "Offer",
                    itemOffered: {
                      "@type": "Service",
                      name: "Transactions Immobilières",
                      description: "Accompagnement pour l'achat et la vente de biens",
                      url: "https://dogesadb.fr/services/transactions",
                    },
                  },
                ],
              },
            }),
          }}
        />
        <style>
          {`
    /* Reset margins and paddings */
    body, html {
      margin: 0 !important;
      padding: 0 !important;
      overflow-x: hidden;
    }
    
    /* Styles pour forcer la transparence du header */
    header {
      background: transparent !important;
      backdrop-filter: blur(10px) !important;
      -webkit-backdrop-filter: blur(10px) !important;
      border-bottom: 1px solid rgba(255, 255, 255, 0.1) !important;
      margin-top: 0 !important;
      padding-top: 0 !important;
    }
    
    /* Suppression de tout élément qui pourrait ajouter un fond */
    header::before,
    header::after {
      display: none !important;
      content: none !important;
      background: none !important;
    }
    
    /* Assurez-vous que le corps de la page commence bien sous le header */
    main {
      padding-top: 0 !important;
      margin-top: 0 !important;
    }

    /* Styles pour le chatbot - priorité absolue */
    .chatbot-widget {
      position: fixed !important;
      bottom: 1rem !important;
      right: 1rem !important;
      z-index: 999999 !important;
      pointer-events: auto !important;
    }
  `}
        </style>
      </head>
      <body className={`${inter.className} m-0 p-0 overflow-x-hidden`}>
        <ScrollToTop />
        <SiteHeader />
        <main className="pt-0">{children}</main>
        <SiteFooter />
        <div className="chatbot-widget">
          <IntelligentChatbot />
        </div>
      </body>
    </html>
  )
}
