import type { Metadata } from "next"
import Image from "next/image"
import { Check } from "lucide-react"
import ServiceHeader from "@/components/service-header"
import CTASection from "@/components/cta-section"
import SmartContactForm from "@/components/smart-contact-form"

export const metadata: Metadata = {
  title: "Administrateur de biens Hauts-de-Seine (92) | DOGES",
  description:
    "Services d'administration de biens immobiliers dans les Hauts-de-Seine. Gestion locative, syndic de copropriété et conseil patrimonial par des experts locaux.",
  keywords:
    "administrateur de biens Hauts-de-Seine, gestion immobilière 92, syndic copropriété Hauts-de-Seine, gestionnaire patrimonial Neuilly Levallois Boulogne, DOGES",
}

export default function AdministrateurBiensHautsDeSeinePage() {
  return (
    <>
      <ServiceHeader
        title="Administrateur de biens dans les Hauts-de-Seine (92)"
        description="Une gestion professionnelle pour votre patrimoine immobilier dans l'ouest parisien"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Administrateur de biens en Île-de-France", url: "/services/administrateur-biens-ile-de-france" },
          { name: "Administrateur de biens Hauts-de-Seine", url: "/services/administrateur-biens-hauts-de-seine" },
        ]}
        backgroundImage="https://images.unsplash.com/photo-1449824913935-59a10b8d2000?q=80&w=2070&auto=format&fit=crop"
      />

      {/* Présentation du service */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Une expertise complète en administration de biens dans les Hauts-de-Seine
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                DOGES propose des services d'administration de biens sur l'ensemble du département des Hauts-de-Seine,
                adaptés aux spécificités de chaque commune. Nos équipes d'experts locaux assurent une gestion de
                proximité efficace et personnalisée.
              </p>
              <p className="text-lg text-gray-700 mb-6">
                Qu'il s'agisse de gestion locative, de syndic de copropriété ou de conseil patrimonial, nous intervenons
                dans toutes les communes des Hauts-de-Seine avec la même exigence de qualité et de professionnalisme.
              </p>
              <div className="space-y-4">
                {[
                  "Services complets d'administration de biens sur tout le département des Hauts-de-Seine",
                  "Équipes spécialisées par secteur pour une expertise locale précise",
                  "Gestion adaptée aux spécificités des marchés immobiliers de chaque commune",
                  "Solutions sur mesure pour tous types de patrimoine (résidentiel, commercial, mixte)",
                  "Veille réglementaire locale pour une conformité totale",
                  "Suivi personnalisé avec un interlocuteur dédié",
                ].map((point, index) => (
                  <div key={index} className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>{point}</p>
                  </div>
                ))}
              </div>
            </div>
            <div className="relative h-[500px] rounded-2xl overflow-hidden">
              <Image
                src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?q=80&w=2070&auto=format&fit=crop"
                alt="Administrateur de biens Hauts-de-Seine DOGES"
                fill
                className="object-cover"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Nos implantations */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
            Nos services d'administration de biens dans les principales communes des Hauts-de-Seine
          </h2>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8 max-w-6xl mx-auto">
            {[
              {
                commune: "Neuilly-sur-Seine",
                description:
                  "Administration de biens dans cette commune résidentielle prisée, avec une expertise des spécificités du marché immobilier haut de gamme.",
                lien: "/services/administrateur-biens-neuilly-sur-seine",
              },
              {
                commune: "Levallois-Perret",
                description:
                  "Gestion immobilière adaptée au dynamisme et à l'attractivité de cette commune en plein essor.",
                lien: "/services/administrateur-biens-levallois-perret",
              },
              {
                commune: "Boulogne-Billancourt",
                description:
                  "Services d'administration sur mesure pour la plus grande ville du département, alliant résidentiel et tertiaire.",
                lien: "/services/administrateur-biens-boulogne-billancourt",
              },
              {
                commune: "Courbevoie",
                description:
                  "Gestion de proximité pour tous types de biens dans cette commune dynamique abritant La Défense.",
                lien: "/services/administrateur-biens-courbevoie",
              },
              {
                commune: "Puteaux",
                description:
                  "Administration de biens adaptée aux spécificités de cette commune mixte entre quartiers d'affaires et résidentiels.",
                lien: "/services/administrateur-biens-puteaux",
              },
              {
                commune: "Issy-les-Moulineaux",
                description:
                  "Services personnalisés pour les propriétaires et copropriétés de cette ville innovante du sud du département.",
                lien: "/services/administrateur-biens-issy-les-moulineaux",
              },
              {
                commune: "Rueil-Malmaison",
                description:
                  "Gestion immobilière sur mesure pour cette commune résidentielle verdoyante de l'ouest des Hauts-de-Seine.",
                lien: "/services/administrateur-biens-rueil-malmaison",
              },
              {
                commune: "Suresnes",
                description:
                  "Administration de biens adaptée aux caractéristiques de cette commune entre ville et nature.",
                lien: "/services/administrateur-biens-suresnes",
              },
              {
                commune: "Nanterre",
                description: "Services complets pour la préfecture du département, en pleine transformation urbaine.",
                lien: "/services/administrateur-biens-nanterre",
              },
            ].map((ville, index) => (
              <div key={index} className="bg-white rounded-xl shadow-md p-6 flex flex-col h-full">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">{ville.commune}</h3>
                <p className="text-gray-700 mb-4 flex-grow">{ville.description}</p>
                <a href={ville.lien} className="text-[#002395] font-medium hover:underline">
                  En savoir plus →
                </a>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Nos services */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
            Une offre complète de services d'administration de biens dans les Hauts-de-Seine
          </h2>

          <div className="grid md:grid-cols-3 gap-8 max-w-5xl mx-auto">
            {[
              {
                title: "Gestion locative",
                services: [
                  "Recherche et sélection des locataires",
                  "Rédaction des baux et état des lieux",
                  "Encaissement et répartition des loyers",
                  "Gestion des charges et régularisations",
                  "Suivi des travaux et entretiens",
                  "Gestion des sinistres et litiges",
                ],
              },
              {
                title: "Syndic de copropriété",
                services: [
                  "Organisation des assemblées générales",
                  "Mise en œuvre des décisions votées",
                  "Gestion administrative et comptable",
                  "Gestion des contrats et fournisseurs",
                  "Suivi des travaux et maintenances",
                  "Conseil technique et juridique",
                ],
              },
              {
                title: "Conseil patrimonial",
                services: [
                  "Analyse et stratégie d'investissement",
                  "Optimisation fiscale immobilière",
                  "Montage de SCI et autres structures",
                  "Conseil en transmission de patrimoine",
                  "Études de rentabilité locative",
                  "Accompagnement projets d'acquisition",
                ],
              },
            ].map((offre, index) => (
              <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                <h3 className="text-xl font-bold mb-6 text-center text-[#002395]">{offre.title}</h3>
                <ul className="space-y-3">
                  {offre.services.map((service, idx) => (
                    <li key={idx} className="flex items-start">
                      <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                        <Check className="h-4 w-4 text-white" />
                      </div>
                      <p>{service}</p>
                    </li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Spécificités du marché des Hauts-de-Seine */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto">
            <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
              Notre expertise du marché immobilier des Hauts-de-Seine
            </h2>

            <div className="grid md:grid-cols-2 gap-8">
              {[
                {
                  title: "Un marché immobilier premium",
                  description:
                    "Notre connaissance approfondie du marché haut de gamme des Hauts-de-Seine nous permet d'optimiser la gestion et la valorisation de votre patrimoine dans ce secteur prisé.",
                },
                {
                  title: "Mixité résidentielle et tertiaire",
                  description:
                    "Nous maîtrisons les spécificités de la gestion mixte, particulièrement importante dans un département qui abrite à la fois des zones résidentielles et le premier quartier d'affaires européen.",
                },
                {
                  title: "Copropriétés haut de gamme",
                  description:
                    "Notre expertise en gestion de syndic s'adapte parfaitement aux exigences des copropriétés prestigieuses qui caractérisent de nombreuses communes du département.",
                },
                {
                  title: "Réglementations locales",
                  description:
                    "Nous assurons une veille permanente des réglementations spécifiques aux Hauts-de-Seine pour garantir une conformité totale de votre gestion immobilière.",
                },
                {
                  title: "Valorisation patrimoniale",
                  description:
                    "Notre connaissance des tendances du marché local nous permet de vous conseiller efficacement pour optimiser la valorisation de votre patrimoine sur le long terme.",
                },
                {
                  title: "Réseau d'intervenants qualifiés",
                  description:
                    "Nous avons développé un réseau d'artisans et prestataires de qualité dans tout le département pour des interventions rapides et à la hauteur des exigences du marché local.",
                },
              ].map((point, index) => (
                <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                  <h3 className="text-lg font-semibold mb-2">{point.title}</h3>
                  <p className="text-gray-700">{point.description}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Pourquoi DOGES */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div className="order-2 lg:order-1">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Pourquoi choisir DOGES comme administrateur de biens dans les Hauts-de-Seine ?
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                Avec plus de 25 ans d'expérience dans l'administration de biens dans les Hauts-de-Seine, DOGES s'impose
                comme le partenaire idéal pour la gestion de votre patrimoine immobilier.
              </p>
              <div className="space-y-6">
                {[
                  {
                    title: "Expertise locale inégalée",
                    description:
                      "Notre présence historique dans les Hauts-de-Seine nous confère une connaissance fine des particularités de chaque commune et quartier du département.",
                  },
                  {
                    title: "Équipes spécialisées par secteur",
                    description:
                      "Chaque propriétaire bénéficie d'un gestionnaire expert de son secteur géographique pour un accompagnement parfaitement adapté aux réalités locales.",
                  },
                  {
                    title: "Transparence et reporting détaillé",
                    description:
                      "Notre plateforme digitale vous donne accès en temps réel à toutes les informations concernant la gestion de votre bien, où que vous soyez.",
                  },
                  {
                    title: "Solutions sur mesure",
                    description:
                      "Nos services s'adaptent à la taille de votre patrimoine et à vos objectifs spécifiques, que vous possédiez un studio ou plusieurs immeubles dans le département.",
                  },
                  {
                    title: "Qualité de service certifiée",
                    description:
                      "Notre engagement pour l'excellence nous a permis d'obtenir des certifications qui garantissent la qualité de nos prestations sur l'ensemble du territoire des Hauts-de-Seine.",
                  },
                ].map((avantage, index) => (
                  <div key={index} className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <div>
                      <h3 className="font-semibold">{avantage.title}</h3>
                      <p className="text-gray-700">{avantage.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
            <div className="order-1 lg:order-2 relative h-[500px] rounded-2xl overflow-hidden">
              <Image
                src="https://images.unsplash.com/photo-1560520653-9e0e4c89eb11?q=80&w=2073&auto=format&fit=crop"
                alt="Administrateur de biens Hauts-de-Seine DOGES"
                fill
                className="object-cover"
              />
            </div>
          </div>
        </div>
      </section>

      <CTASection
        title="Confiez l'administration de vos biens à nos experts des Hauts-de-Seine"
        description="Bénéficiez d'un accompagnement personnalisé pour la gestion de votre patrimoine immobilier dans les Hauts-de-Seine."
        buttonText="Demander un devis gratuit"
        buttonLink="/contact"
      />

      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl mx-auto">
            <h2 className="text-3xl font-bold mb-8 text-center text-[#002395]">
              Contactez notre équipe d'administrateurs de biens des Hauts-de-Seine
            </h2>
            <p className="text-lg text-gray-700 mb-8 text-center">
              Nos experts sont à votre disposition pour répondre à vos questions et vous proposer une solution
              d'administration adaptée à votre patrimoine dans les Hauts-de-Seine.
            </p>
            <SmartContactForm />
          </div>
        </div>
      </section>
    </>
  )
}
