import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check } from "lucide-react"
import ServiceHeader from "@/components/service-header"
import CTASection from "@/components/cta-section"
import SmartContactForm from "@/components/smart-contact-form"

export const metadata: Metadata = {
  title: "Administrateur de biens à Paris (75) | DOGES",
  description:
    "Services d'administration de biens immobiliers à Paris. Gestion locative, syndic de copropriété et conseil patrimonial par des experts locaux.",
  keywords:
    "administrateur de biens Paris, gestion immobilière 75, syndic copropriété Paris, gestionnaire patrimonial Paris, DOGES",
}

export default function AdministrateurBiensParisPage() {
  return (
    <>
      <ServiceHeader
        title="Administrateur de biens à Paris (75)"
        description="Une gestion professionnelle pour votre patrimoine immobilier parisien"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Administrateur de biens en Île-de-France", url: "/services/administrateur-biens-ile-de-france" },
          { name: "Paris", url: "/services/administrateur-biens-paris" },
        ]}
        backgroundImage="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/image-hiLC5FUeE7lMk590ygT1lnPRnBPLU7.png"
      />

      {/* Présentation du service */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Une expertise complète en administration de biens à Paris
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                DOGES propose des services d'administration de biens sur l'ensemble des 20 arrondissements parisiens,
                adaptés aux spécificités de chaque quartier. Nos équipes d'experts locaux assurent une gestion de
                proximité efficace et personnalisée.
              </p>
              <p className="text-lg text-gray-700 mb-6">
                Qu'il s'agisse de gestion locative, de syndic de copropriété ou de conseil patrimonial, nous intervenons
                dans tous les arrondissements de Paris avec la même exigence de qualité et de professionnalisme.
              </p>
              <div className="space-y-4">
                {[
                  "Services complets d'administration de biens sur les 20 arrondissements",
                  "Équipes spécialisées par arrondissement pour une expertise locale précise",
                  "Gestion adaptée aux spécificités du marché immobilier parisien",
                  "Solutions sur mesure pour tous types de patrimoine (résidentiel, commercial, mixte)",
                  "Veille réglementaire locale pour une conformité totale",
                  "Suivi personnalisé avec un interlocuteur dédié",
                ].map((point, index) => (
                  <div key={index} className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>{point}</p>
                  </div>
                ))}
              </div>
            </div>
            <div className="relative h-[500px] rounded-2xl overflow-hidden">
              <Image
                src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/image-CcTQSc7CjeZi446vvpQYcwidiXy8jr.png"
                alt="Administrateur de biens à Paris DOGES"
                fill
                className="object-cover"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Nos implantations */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
            Nos services d'administration de biens dans les arrondissements parisiens
          </h2>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8 max-w-6xl mx-auto">
            {[
              {
                zone: "Paris Centre (1er, 2e, 3e, 4e)",
                description:
                  "Administration de biens dans le cœur historique de Paris, avec une expertise des immeubles anciens et des biens d'exception.",
                lien: "/services/administrateur-biens-paris-centre",
              },
              {
                zone: "Paris 5e, 6e, 7e",
                description:
                  "Gestion immobilière dans les arrondissements prestigieux de la Rive Gauche, adaptée aux exigences d'une clientèle haut de gamme.",
                lien: "/services/administrateur-biens-paris-5-6-7",
              },
              {
                zone: "Paris 8e, 16e, 17e",
                description:
                  "Services d'administration sur mesure pour les quartiers résidentiels de l'Ouest parisien et leurs immeubles de standing.",
                lien: "/services/administrateur-biens-paris-8-16-17",
              },
              {
                zone: "Paris 9e, 10e, 18e",
                description:
                  "Gestion de proximité pour ces arrondissements en pleine mutation, entre tradition et dynamisme urbain.",
                lien: "/services/administrateur-biens-paris-9-10-18",
              },
              {
                zone: "Paris 11e, 12e, 20e",
                description:
                  "Administration de biens adaptée à l'Est parisien, avec ses quartiers diversifiés et son marché immobilier en évolution.",
                lien: "/services/administrateur-biens-paris-11-12-20",
              },
              {
                zone: "Paris 13e, 14e, 15e",
                description:
                  "Services personnalisés pour le Sud parisien, entre quartiers familiaux et zones en développement.",
                lien: "/services/administrateur-biens-paris-13-14-15",
              },
              {
                zone: "Paris 19e",
                description: "Gestion immobilière sur mesure pour cet arrondissement en pleine transformation urbaine.",
                lien: "/services/administrateur-biens-paris-19",
              },
              {
                zone: "Immeubles haussmanniens",
                description:
                  "Expertise spécifique pour la gestion des immeubles haussmanniens et leurs problématiques particulières.",
                lien: "/services/administrateur-biens-immeubles-haussmanniens",
              },
            ].map((zone, index) => (
              <div key={index} className="bg-white rounded-xl shadow-md p-6 flex flex-col h-full">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">{zone.zone}</h3>
                <p className="text-gray-700 mb-4 flex-grow">{zone.description}</p>
                <Link href={zone.lien} className="text-[#002395] font-medium hover:underline">
                  En savoir plus →
                </Link>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Nos services */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
            Une offre complète de services d'administration de biens à Paris
          </h2>

          <div className="grid md:grid-cols-3 gap-8 max-w-5xl mx-auto">
            {[
              {
                title: "Gestion locative",
                services: [
                  "Recherche et sélection des locataires",
                  "Rédaction des baux et état des lieux",
                  "Encaissement et répartition des loyers",
                  "Gestion des charges et régularisations",
                  "Suivi des travaux et entretiens",
                  "Gestion des sinistres et litiges",
                ],
              },
              {
                title: "Syndic de copropriété",
                services: [
                  "Organisation des assemblées générales",
                  "Mise en œuvre des décisions votées",
                  "Gestion administrative et comptable",
                  "Gestion des contrats et fournisseurs",
                  "Suivi des travaux et maintenances",
                  "Conseil technique et juridique",
                ],
              },
              {
                title: "Conseil patrimonial",
                services: [
                  "Analyse et stratégie d'investissement",
                  "Optimisation fiscale immobilière",
                  "Montage de SCI et autres structures",
                  "Conseil en transmission de patrimoine",
                  "Études de rentabilité locative",
                  "Accompagnement projets d'acquisition",
                ],
              },
            ].map((offre, index) => (
              <div key={index} className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
                <h3 className="text-xl font-bold mb-6 text-center text-[#002395]">{offre.title}</h3>
                <ul className="space-y-3">
                  {offre.services.map((service, idx) => (
                    <li key={idx} className="flex items-start">
                      <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                        <Check className="h-4 w-4 text-white" />
                      </div>
                      <p>{service}</p>
                    </li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Spécificités du marché immobilier parisien */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto">
            <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
              Notre expertise du marché immobilier parisien
            </h2>

            <div className="grid md:grid-cols-2 gap-8">
              {[
                {
                  title: "Un marché immobilier d'exception",
                  description:
                    "Paris reste l'un des marchés immobiliers les plus prisés au monde, avec des spécificités que nos experts maîtrisent parfaitement.",
                },
                {
                  title: "Encadrement des loyers",
                  description:
                    "Notre gestion locative intègre parfaitement les contraintes de l'encadrement des loyers parisien pour une conformité totale.",
                },
                {
                  title: "Immeubles anciens et patrimoniaux",
                  description:
                    "Notre expertise inclut la gestion des immeubles historiques et patrimoniaux, avec leurs problématiques spécifiques.",
                },
                {
                  title: "Copropriétés complexes",
                  description:
                    "Nous gérons de nombreuses copropriétés parisiennes, souvent complexes, avec des services adaptés aux attentes élevées des copropriétaires.",
                },
                {
                  title: "Investissements locatifs performants",
                  description:
                    "Notre connaissance du marché locatif parisien vous permet d'optimiser vos investissements et d'en maximiser la rentabilité.",
                },
                {
                  title: "Réglementations spécifiques",
                  description:
                    "Nous assurons une veille permanente sur les réglementations parisiennes (locations touristiques, DPE, etc.) pour une gestion conforme.",
                },
              ].map((point, index) => (
                <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                  <h3 className="text-lg font-semibold mb-2">{point.title}</h3>
                  <p className="text-gray-700">{point.description}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Pourquoi DOGES */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div className="order-2 lg:order-1">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Pourquoi choisir DOGES comme administrateur de biens à Paris ?
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                Avec plus de 25 ans d'expérience dans l'administration de biens à Paris, DOGES s'impose comme le
                partenaire idéal pour la gestion de votre patrimoine immobilier parisien.
              </p>
              <div className="space-y-6">
                {[
                  {
                    title: "Expertise parisienne inégalée",
                    description:
                      "Notre présence historique à Paris nous confère une connaissance fine des particularités de chaque arrondissement et quartier.",
                  },
                  {
                    title: "Équipes spécialisées par secteur",
                    description:
                      "Chaque propriétaire bénéficie d'un gestionnaire expert de son secteur géographique pour un accompagnement parfaitement adapté.",
                  },
                  {
                    title: "Transparence et reporting détaillé",
                    description:
                      "Notre plateforme digitale vous donne accès en temps réel à toutes les informations concernant la gestion de votre bien, où que vous soyez.",
                  },
                  {
                    title: "Solutions sur mesure",
                    description:
                      "Nos services s'adaptent à la taille de votre patrimoine et à vos objectifs spécifiques, que vous possédiez un studio ou plusieurs immeubles.",
                  },
                  {
                    title: "Réseau de partenaires locaux",
                    description:
                      "Nous collaborons avec des artisans et prestataires qualifiés dans tout Paris pour des interventions rapides et de qualité.",
                  },
                ].map((avantage, index) => (
                  <div key={index} className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <div>
                      <h3 className="font-semibold">{avantage.title}</h3>
                      <p className="text-gray-700">{avantage.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
            <div className="order-1 lg:order-2 relative h-[500px] rounded-2xl overflow-hidden">
              <Image
                src="https://hebbkx1anhila5yf.public.blob.vercel-storage.com/image-FodJXHv0wxbBGGXMP3sTZRN2JKgqaP.png"
                alt="Gestion immobilière à Paris"
                fill
                className="object-cover"
              />
            </div>
          </div>
        </div>
      </section>

      <CTASection
        title="Confiez l'administration de vos biens à nos experts parisiens"
        description="Bénéficiez d'un accompagnement personnalisé pour la gestion de votre patrimoine immobilier à Paris."
        buttonText="Demander un devis gratuit"
        buttonLink="/contact"
      />

      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl mx-auto">
            <h2 className="text-3xl font-bold mb-8 text-center text-[#002395]">
              Contactez notre équipe d'administrateurs de biens parisiens
            </h2>
            <p className="text-lg text-gray-700 mb-8 text-center">
              Nos experts sont à votre disposition pour répondre à vos questions et vous proposer une solution
              d'administration adaptée à votre patrimoine parisien.
            </p>
            <SmartContactForm />
          </div>
        </div>
      </section>
    </>
  )
}
