import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check } from "lucide-react"
import ServiceHeader from "@/components/service-header"
import CTASection from "@/components/cta-section"
import SmartContactForm from "@/components/smart-contact-form"
import  TableOfContentsWidget  from "@/components/table-of-contents-widget"
import { ContactWidget } from "@/components/contact-widget"
import RelatedServicesWidget  from "@/components/related-services-widget"

export const metadata: Metadata = {
  title: "Administrateur de biens Val-d'Oise (95) | DOGES",
  description:
    "Services d'administration de biens immobiliers dans le Val-d'Oise (95). Gestion locative, syndic de copropriété et conseil patrimonial par des experts locaux.",
  keywords:
    "administrateur de biens Val-d'Oise, gestion immobilière 95, syndic copropriété Cergy, Argenteuil, Sarcelles, Pontoise, DOGES",
}

export default function AdministrateurBiensValDoisePage() {
  const tableOfContents = [
    {
      id: "presentation",
      title: "Notre expertise dans le Val-d'Oise",
    },
    {
      id: "services",
      title: "Nos services d'administration",
    },
    {
      id: "communes",
      title: "Communes du Val-d'Oise",
    },
    {
      id: "avantages",
      title: "Nos avantages",
    },
    {
      id: "contact",
      title: "Nous contacter",
    },
  ]

  return (
    <>
      <div className="w-full">
        <ServiceHeader
          title="Administrateur de biens dans le Val-d'Oise (95)"
          description="Une gestion professionnelle pour votre patrimoine immobilier dans le Val-d'Oise"
          breadcrumbItems={[
            { name: "Accueil", url: "/" },
            { name: "Services", url: "/services" },
            { name: "Administrateur de biens en Île-de-France", url: "/services/administrateur-biens-ile-de-france" },
            { name: "Administrateur de biens Val-d'Oise", url: "/services/administrateur-biens-val-doise" },
          ]}
          backgroundImage="https://images.unsplash.com/photo-1449824913935-59a10b8d2000?q=80&w=2070&auto=format&fit=crop"
        />
      </div>

      <div className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">
            {/* Présentation */}
            <section id="presentation" className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Notre expertise d'administrateur de biens dans le Val-d'Oise
              </h2>
              <div className="grid md:grid-cols-2 gap-8 items-center mb-8">
                <div>
                  <p className="text-gray-700 mb-4">
                    DOGES propose des services d'administration de biens sur l'ensemble du Val-d'Oise (95), adaptés aux
                    spécificités de ce département dynamique du nord de l'Île-de-France. Nos équipes d'experts locaux
                    assurent une gestion de proximité efficace et personnalisée.
                  </p>
                  <p className="text-gray-700 mb-4">
                    Que vous possédiez un bien à Cergy, Argenteuil, Pontoise, Sarcelles ou dans toute autre commune du
                    département, nous vous offrons une gestion immobilière complète et sur mesure, tenant compte des
                    particularités du marché local.
                  </p>
                  <div className="space-y-2">
                    {[
                      "Connaissance approfondie du marché immobilier du Val-d'Oise",
                      "Équipe dédiée aux spécificités du département",
                      "Suivi personnalisé avec un interlocuteur unique",
                      "Présence locale pour une réactivité optimale",
                    ].map((point, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p className="text-gray-700">{point}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-64 rounded-lg overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1486406146926-c627a92ad1ab?q=80&w=2070&auto=format&fit=crop"
                    alt="Gestion immobilière dans le Val-d'Oise"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Nos services */}
            <section id="services" className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Nos services d'administration de biens dans le Val-d'Oise
              </h2>
              <div className="grid md:grid-cols-3 gap-6 mb-8">
                {[
                  {
                    title: "Gestion locative",
                    services: [
                      "Recherche et sélection des locataires",
                      "Rédaction des baux et état des lieux",
                      "Encaissement et répartition des loyers",
                      "Gestion des charges et régularisations",
                      "Suivi des travaux et entretiens",
                      "Gestion des sinistres et litiges",
                    ],
                  },
                  {
                    title: "Syndic de copropriété",
                    services: [
                      "Organisation des assemblées générales",
                      "Mise en œuvre des décisions votées",
                      "Gestion administrative et comptable",
                      "Gestion des contrats et fournisseurs",
                      "Suivi des travaux et maintenances",
                      "Conseil technique et juridique",
                    ],
                  },
                  {
                    title: "Conseil patrimonial",
                    services: [
                      "Analyse et stratégie d'investissement",
                      "Optimisation fiscale immobilière",
                      "Montage de SCI et autres structures",
                      "Conseil en transmission de patrimoine",
                      "Études de rentabilité locative",
                      "Accompagnement projets d'acquisition",
                    ],
                  },
                ].map((service, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-md p-6 border border-gray-100">
                    <h3 className="text-xl font-semibold mb-4 text-[#002395]">{service.title}</h3>
                    <ul className="space-y-2">
                      {service.services.map((item, idx) => (
                        <li key={idx} className="flex items-start">
                          <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                            <Check className="h-3 w-3 text-white" />
                          </div>
                          <p className="text-sm text-gray-700">{item}</p>
                        </li>
                      ))}
                    </ul>
                  </div>
                ))}
              </div>
            </section>

            {/* Communes */}
            <section id="communes" className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Nos services d'administrateur de biens dans les communes du Val-d'Oise
              </h2>
              <p className="text-gray-700 mb-6">
                DOGES intervient dans l'ensemble des communes du Val-d'Oise (95). Nos équipes connaissent parfaitement
                les spécificités de chaque secteur et vous proposent une gestion adaptée à votre localisation.
              </p>

              <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-4 mb-8">
                {[
                  "Cergy",
                  "Argenteuil",
                  "Sarcelles",
                  "Pontoise",
                  "Garges-lès-Gonesse",
                  "Franconville",
                  "Ermont",
                  "Gonesse",
                  "Goussainville",
                  "Herblay",
                  "Bezons",
                  "Eaubonne",
                  "Taverny",
                  "Montmorency",
                  "Saint-Ouen-l'Aumône",
                  "Cormeilles-en-Parisis",
                  "Deuil-la-Barre",
                  "Sannois",
                  "Villiers-le-Bel",
                  "Saint-Gratien",
                  "Montigny-lès-Cormeilles",
                  "Soisy-sous-Montmorency",
                  "Enghien-les-Bains",
                  "L'Isle-Adam",
                ].map((commune, index) => (
                  <div
                    key={index}
                    className="bg-white border border-gray-200 rounded-md px-4 py-3 hover:bg-gray-50 transition-colors"
                  >
                    <Link href={`/services/administrateur-biens-${commune.toLowerCase().replace(/['\s-]/g, "-")}`}>
                      <span className="text-[#002395] hover:underline">Administrateur de biens {commune}</span>
                    </Link>
                  </div>
                ))}
              </div>
            </section>

            {/* Avantages */}
            <section id="avantages" className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Pourquoi choisir DOGES comme administrateur de biens dans le Val-d'Oise ?
              </h2>
              <div className="grid md:grid-cols-2 gap-6 mb-8">
                <div className="relative h-64 rounded-lg overflow-hidden">
                  <Image
                    src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?q=80&w=2070&auto=format&fit=crop"
                    alt="Équipe d'administrateurs de biens DOGES dans le Val-d'Oise"
                    fill
                    className="object-cover"
                  />
                </div>
                <div className="space-y-4">
                  {[
                    {
                      title: "Expertise locale inégalée",
                      description:
                        "Notre présence dans le Val-d'Oise nous confère une connaissance fine des particularités du marché immobilier local.",
                    },
                    {
                      title: "Équipe dédiée au territoire",
                      description:
                        "Chaque propriétaire bénéficie d'un gestionnaire expert du Val-d'Oise pour un accompagnement parfaitement adapté.",
                    },
                    {
                      title: "Transparence et reporting détaillé",
                      description:
                        "Notre plateforme digitale vous donne accès en temps réel à toutes les informations concernant la gestion de votre bien.",
                    },
                    {
                      title: "Solutions sur mesure",
                      description:
                        "Nos services s'adaptent à la taille de votre patrimoine et à vos objectifs spécifiques, que vous possédiez un studio ou plusieurs immeubles.",
                    },
                  ].map((avantage, index) => (
                    <div key={index} className="flex items-start">
                      <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                        <Check className="h-4 w-4 text-white" />
                      </div>
                      <div>
                        <h3 className="font-semibold">{avantage.title}</h3>
                        <p className="text-gray-700">{avantage.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* Spécificités du marché */}
            <section className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Les spécificités du marché immobilier du Val-d'Oise
              </h2>
              <p className="text-gray-700 mb-4">
                Le Val-d'Oise présente un marché immobilier diversifié, entre zones urbaines denses, secteurs
                pavillonnaires et espaces plus ruraux. Notre expertise nous permet d'adapter notre gestion à ces
                différentes réalités :
              </p>
              <div className="grid md:grid-cols-2 gap-6 mb-8">
                {[
                  {
                    title: "Secteur de Cergy-Pontoise",
                    description:
                      "Pôle économique et universitaire majeur, avec une demande locative soutenue et un potentiel d'investissement intéressant.",
                  },
                  {
                    title: "Zone d'Argenteuil et Bezons",
                    description:
                      "Secteur en mutation, bénéficiant de la proximité de Paris et de La Défense, avec un marché dynamique.",
                  },
                  {
                    title: "Secteur de la Vallée de Montmorency",
                    description:
                      "Zone résidentielle prisée avec un marché immobilier haut de gamme et des copropriétés de standing.",
                  },
                  {
                    title: "Secteur de Roissy et Plaine de France",
                    description:
                      "Marché porté par l'activité économique de l'aéroport, avec une forte demande locative professionnelle.",
                  },
                ].map((secteur, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-5 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{secteur.title}</h3>
                    <p className="text-gray-700">{secteur.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Contact */}
            <section id="contact" className="mb-12">
              <h2 className="text-2xl font-bold mb-6 text-[#002395]">
                Contactez notre équipe d'administrateurs de biens dans le Val-d'Oise
              </h2>
              <p className="text-gray-700 mb-6">
                Nos experts sont à votre disposition pour répondre à vos questions et vous proposer une solution
                d'administration adaptée à votre patrimoine dans le Val-d'Oise.
              </p>
              <SmartContactForm />
            </section>
          </div>

          {/* Barre latérale */}
          <div className="lg:w-1/3 space-y-8">
            <TableOfContentsWidget items={tableOfContents} />
            <ContactWidget />
            <RelatedServicesWidget
              services={[
                {
                  title: "Gestion locative Val-d'Oise",
                  link: "/services/gestion-locative-ile-de-france/val-doise",
                },
                {
                  title: "Syndic de copropriété Val-d'Oise",
                  link: "/services/syndic-copropriete-val-doise",
                },
                {
                  title: "Garantie loyers impayés",
                  link: "/services/garantie-loyers-impayes",
                },
                {
                  title: "Assurance PNO",
                  link: "/services/assurance-pno",
                },
              ]}
            />
          </div>
        </div>
      </div>

      <CTASection
        title="Confiez l'administration de vos biens à nos experts du Val-d'Oise"
        description="Bénéficiez d'un accompagnement personnalisé pour la gestion de votre patrimoine immobilier dans le Val-d'Oise."
        buttonText="Demander un devis gratuit"
        buttonLink="/contact"
      />
    </>
  )
}
