import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { ArrowRight } from "lucide-react"
import { Button } from "@/components/ui/button"
import ServiceHeader from "@/components/service-header"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import RelatedServicesWidget from "@/components/related-services-widget"
import ContactWidget from "@/components/contact-widget"
import { BreadcrumbSchema } from "@/components/schema/breadcrumb-schema"
import { FAQSchema } from "@/components/schema/faq-schema"

export const metadata: Metadata = {
  title: "Comptabilité Mandants | DOGES Administrateur de Biens",
  description:
    "Service de comptabilité mandants pour propriétaires bailleurs à Paris et en Île-de-France. Gestion transparente et précise de vos comptes locatifs.",
}

export default function ComptabiliteMandantsPage() {
  const tableOfContents = [
    {
      title: "Notre service de comptabilité mandants",
      id: "service-comptabilite",
    },
    {
      title: "Avantages de notre comptabilité mandants",
      id: "avantages-comptabilite",
    },
    {
      title: "Transparence et reporting",
      id: "transparence-reporting",
    },
    {
      title: "Optimisation fiscale",
      id: "optimisation-fiscale",
    },
    {
      title: "Tarifs et prestations",
      id: "tarifs-prestations",
    },
    {
      title: "Questions fréquentes",
      id: "faq",
    },
  ]

  const relatedServices = [
    {
      title: "Gestion Locative",
      description: "Service complet de gestion de vos biens immobiliers",
      link: "/services/gestion-locative",
    },
    {
      title: "Garantie Loyers Impayés",
      description: "Protection contre les risques d'impayés",
      link: "/services/garantie-loyers-impayes",
    },
    {
      title: "Assurance PNO",
      description: "Assurance propriétaire non occupant",
      link: "/services/assurance-pno",
    },
  ]

  const faqs = [
    {
      question: "Quels documents comptables me fournissez-vous ?",
      answer:
        "Nous vous fournissons un relevé de gérance mensuel détaillé, un bilan annuel complet, ainsi que tous les documents nécessaires pour votre déclaration fiscale (revenus fonciers, BIC pour location meublée).",
    },
    {
      question: "Comment accéder à mes documents comptables ?",
      answer:
        "Tous vos documents comptables sont disponibles 24h/24 et 7j/7 dans votre espace client sécurisé. Vous recevez également une notification par email lors de la mise à disposition de nouveaux documents.",
    },
    {
      question: "Quand sont effectués les virements des loyers ?",
      answer:
        "Les virements des loyers sont effectués entre le 5 et le 10 de chaque mois, après encaissement des loyers et déduction des charges et de nos honoraires.",
    },
    {
      question: "Comment sont gérées les charges de copropriété ?",
      answer:
        "Nous provisionnons les charges de copropriété selon le budget prévisionnel et procédons à la régularisation annuelle. Tout est détaillé dans votre relevé de gérance.",
    },
    {
      question: "Proposez-vous une aide pour la déclaration fiscale ?",
      answer:
        "Oui, nous vous fournissons tous les documents nécessaires pour votre déclaration fiscale et pouvons vous orienter vers des experts-comptables partenaires pour une optimisation fiscale personnalisée.",
    },
  ]

  return (
    <>
      <BreadcrumbSchema
        items={[
          { name: "Accueil", item: "https://www.doges.fr/" },
          { name: "Services", item: "https://www.doges.fr/services" },
          { name: "Comptabilité Mandants", item: "https://www.doges.fr/services/comptabilite-mandants" },
        ]}
      />

      <FAQSchema
        mainEntity={faqs.map((faq) => ({
          questionName: faq.question,
          acceptedAnswerText: faq.answer,
        }))}
      />

      {/* Header en pleine largeur */}
      <div className="w-full">
        <ServiceHeader
          title="Comptabilité Mandants"
          description="Une gestion comptable transparente et précise pour vos biens immobiliers"
          breadcrumbItems={[
            { name: "Accueil", url: "/" },
            { name: "Services", url: "/services" },
            { name: "Comptabilité Mandants", url: "/services/comptabilite-mandants" },
          ]}
          backgroundImage="/real-estate-accounting-documents.png"
        />
      </div>

      {/* Fil d'Ariane */}
      <div className="w-full bg-gray-100 py-3">
        <div className="container mx-auto px-4">{/* Breadcrumb component would go here */}</div>
      </div>

      {/* Contenu principal avec container */}
      <div className="container mx-auto px-4 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">
            <section id="service-comptabilite" className="mb-12">
              <h2 className="text-2xl font-bold mb-4">Notre service de comptabilité mandants</h2>
              <p className="mb-4">
                Chez DOGES, nous comprenons l'importance d'une gestion comptable rigoureuse et transparente pour vos
                biens immobiliers. Notre service de comptabilité mandants est conçu pour vous offrir une tranquillité
                d'esprit totale concernant la gestion financière de votre patrimoine.
              </p>
              <p className="mb-4">
                Notre équipe de comptables spécialisés en immobilier assure un suivi précis de tous les flux financiers
                liés à vos biens : encaissement des loyers, paiement des charges, provisions pour travaux, et bien plus
                encore.
              </p>
              <p>
                Grâce à nos outils numériques de pointe, vous bénéficiez d'une visibilité complète sur la situation
                financière de vos biens, avec des rapports détaillés et accessibles à tout moment via votre espace
                client sécurisé.
              </p>
            </section>

            <section id="avantages-comptabilite" className="mb-12">
              <h2 className="text-2xl font-bold mb-4">Avantages de notre comptabilité mandants</h2>
              <div className="grid md:grid-cols-2 gap-6 mb-6">
                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Précision et rigueur</h3>
                  <p>
                    Chaque centime est comptabilisé avec précision, garantissant une gestion financière irréprochable de
                    votre patrimoine immobilier.
                  </p>
                </div>
                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Conformité légale</h3>
                  <p>
                    Nos procédures comptables respectent scrupuleusement la législation en vigueur, vous protégeant de
                    tout risque juridique ou fiscal.
                  </p>
                </div>
                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Simplicité administrative</h3>
                  <p>
                    Fini les tracas administratifs ! Nous nous occupons de toute la paperasse comptable, vous permettant
                    de vous concentrer sur l'essentiel.
                  </p>
                </div>
                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Accompagnement fiscal</h3>
                  <p>
                    Nous vous fournissons tous les documents nécessaires pour vos déclarations fiscales et vous
                    conseillons pour optimiser votre situation.
                  </p>
                </div>
              </div>
            </section>

            <section id="transparence-reporting" className="mb-12">
              <h2 className="text-2xl font-bold mb-4">Transparence et reporting</h2>
              <p className="mb-4">
                La transparence est au cœur de notre approche. Nous vous fournissons des rapports comptables clairs et
                détaillés :
              </p>
              <ul className="list-disc pl-6 mb-6 space-y-2">
                <li>
                  <strong>Relevé de gérance mensuel</strong> : détail des loyers perçus, charges payées et solde
                  disponible
                </li>
                <li>
                  <strong>Bilan annuel</strong> : synthèse complète de l'année écoulée avec analyse de rentabilité
                </li>
                <li>
                  <strong>Aide à la déclaration fiscale</strong> : documents préformatés pour faciliter vos déclarations
                </li>
                <li>
                  <strong>Alertes personnalisées</strong> : notification en cas d'anomalie ou de situation nécessitant
                  votre attention
                </li>
              </ul>
              <p>
                Tous ces documents sont accessibles 24h/24 et 7j/7 dans votre espace client sécurisé, et archivés
                pendant toute la durée de notre collaboration.
              </p>
            </section>

            <section id="optimisation-fiscale" className="mb-12">
              <h2 className="text-2xl font-bold mb-4">Optimisation fiscale</h2>
              <div className="relative h-64 mb-6 rounded-lg overflow-hidden">
                <Image
                  src="/real-estate-accounting-documents.png"
                  alt="Document d'optimisation fiscale"
                  fill
                  className="object-cover"
                />
              </div>
              <p className="mb-4">
                Au-delà de la simple comptabilité, nous vous accompagnons dans l'optimisation fiscale de votre
                patrimoine immobilier :
              </p>
              <ul className="list-disc pl-6 mb-6 space-y-2">
                <li>Conseils sur le choix du régime fiscal le plus avantageux (réel ou micro-foncier)</li>
                <li>Optimisation des charges déductibles</li>
                <li>Accompagnement pour les dispositifs de défiscalisation immobilière</li>
                <li>Conseil sur la structuration juridique optimale de votre patrimoine</li>
              </ul>
              <p>
                Notre réseau d'experts-comptables et fiscalistes spécialisés en immobilier peut également vous proposer
                un accompagnement personnalisé pour les situations complexes.
              </p>
            </section>

            <section id="tarifs-prestations" className="mb-12">
              <h2 className="text-2xl font-bold mb-4">Tarifs et prestations</h2>
              <p className="mb-6">
                Notre service de comptabilité mandants est inclus dans nos forfaits de gestion locative. Pour les
                propriétaires souhaitant uniquement bénéficier de ce service, nous proposons des formules adaptées à vos
                besoins :
              </p>
              <div className="overflow-x-auto mb-6">
                <table className="min-w-full bg-white border border-gray-200">
                  <thead>
                    <tr className="bg-gray-50">
                      <th className="py-3 px-4 border-b text-left">Formule</th>
                      <th className="py-3 px-4 border-b text-left">Prestations incluses</th>
                      <th className="py-3 px-4 border-b text-left">Tarif mensuel HT</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr>
                      <td className="py-3 px-4 border-b font-medium">Essentiel</td>
                      <td className="py-3 px-4 border-b">
                        Relevé mensuel, bilan annuel, aide à la déclaration fiscale
                      </td>
                      <td className="py-3 px-4 border-b">À partir de 25€/lot</td>
                    </tr>
                    <tr>
                      <td className="py-3 px-4 border-b font-medium">Premium</td>
                      <td className="py-3 px-4 border-b">
                        Formule Essentiel + analyse de rentabilité, optimisation fiscale
                      </td>
                      <td className="py-3 px-4 border-b">À partir de 35€/lot</td>
                    </tr>
                    <tr>
                      <td className="py-3 px-4 border-b font-medium">Sur-mesure</td>
                      <td className="py-3 px-4 border-b">Prestations personnalisées selon vos besoins spécifiques</td>
                      <td className="py-3 px-4 border-b">Sur devis</td>
                    </tr>
                  </tbody>
                </table>
              </div>
              <p>
                Pour obtenir un devis personnalisé adapté à votre situation, n'hésitez pas à nous contacter. Nous vous
                proposerons la solution la plus adaptée à vos besoins et à votre budget.
              </p>
            </section>

            <section id="faq" className="mb-12">
              <h2 className="text-2xl font-bold mb-6">Questions fréquentes</h2>
              <div className="space-y-6">
                {faqs.map((faq, index) => (
                  <div key={index} className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                    <h3 className="text-xl font-semibold mb-2">{faq.question}</h3>
                    <p>{faq.answer}</p>
                  </div>
                ))}
              </div>
            </section>

            <div className="bg-gray-50 p-8 rounded-lg border border-gray-100 mb-8">
              <h2 className="text-2xl font-bold mb-4">Prêt à simplifier la gestion comptable de vos biens ?</h2>
              <p className="mb-6">
                Contactez-nous dès aujourd'hui pour découvrir comment notre service de comptabilité mandants peut vous
                faire gagner du temps et optimiser la rentabilité de votre patrimoine immobilier.
              </p>
              <Link href="/contact">
                <Button size="lg" className="bg-[#002395] hover:bg-[#001875] text-white">
                  Demander un devis gratuit
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Button>
              </Link>
            </div>
          </div>

          {/* Sidebar */}
          <div className="lg:w-1/3 space-y-6">
            <TableOfContentsWidget items={tableOfContents} />
            <RelatedServicesWidget services={relatedServices} />
            <ContactWidget />
          </div>
        </div>
      </div>
    </>
  )
}
