import { Shield, Check, FileText, Banknote, Scale, Clock } from "lucide-react"
import Image from "next/image"
import type { Metadata } from "next"

import ServiceHeader from "@/components/service-header"
import SmartContactForm from "@/components/smart-contact-form"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import ContactWidget from "@/components/contact-widget"
import RelatedServicesWidget from "@/components/related-services-widget"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Garantie Loyers Impayés (GLI) | DOGES Immobilier",
  description:
    "Protégez vos revenus locatifs avec notre garantie loyers impayés. Couverture complète contre les impayés, dégradations et frais de procédure.",
  keywords: "garantie loyers impayés, GLI, assurance loyers impayés, protection propriétaire, impayés locatifs",
}

export default function GarantieLoyers() {
  return (
    <>
      <ServiceHeader
        title="Garantie Loyers Impayés"
        description="Sécurisez vos revenus locatifs et protégez votre investissement immobilier"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Garantie Loyers Impayés", url: "/services/garantie-loyers-impayes" },
        ]}
        backgroundImage="/rental-income-protection-security.png"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Contenu principal */}
          <div className="lg:col-span-2">
            {/* Section d'introduction */}
            <section className="mb-12" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Qu'est-ce que la Garantie Loyers Impayés ?</h2>
              <div className="grid md:grid-cols-2 gap-8 items-center mb-8">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    La Garantie Loyers Impayés (GLI) est une protection essentielle pour tout propriétaire bailleur.
                    Elle vous garantit le versement des loyers en cas de défaillance du locataire et vous protège contre
                    les risques locatifs.
                  </p>
                  <p className="text-lg text-gray-700">
                    Chez DOGES, nous proposons une solution d'assurance complète pour sécuriser vos revenus locatifs et
                    vous permettre de profiter sereinement de votre investissement immobilier.
                  </p>
                </div>
                <div className="relative h-64 rounded-xl overflow-hidden shadow-lg">
                  <Image
                    src="/rental-income-protection-security.png"
                    alt="Garantie loyers impayés"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>

              <div className="bg-gray-50 p-6 rounded-xl border border-gray-100 mb-8">
                <h3 className="text-xl font-semibold mb-4">Les points clés de notre garantie</h3>
                <div className="grid md:grid-cols-2 gap-4">
                  {[
                    "Prise en charge des loyers impayés jusqu'à 24 mois",
                    "Couverture des frais de procédure juridique",
                    "Indemnisation des dégradations locatives",
                    "Protection juridique complète",
                    "Garantie du maintien de vos revenus locatifs",
                    "Accompagnement personnalisé",
                  ].map((feature, index) => (
                    <div key={index} className="flex items-start">
                      <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                        <Check className="h-4 w-4 text-white" />
                      </div>
                      <p>{feature}</p>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* Section des avantages */}
            <section className="mb-12" id="avantages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Les avantages de notre Garantie Loyers Impayés</h2>
              <div className="grid md:grid-cols-3 gap-6">
                {[
                  {
                    title: "Sécurité financière",
                    description:
                      "Garantie du paiement des loyers même en cas de défaillance du locataire, préservant ainsi vos revenus locatifs.",
                    icon: <Shield className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Tranquillité d'esprit",
                    description:
                      "Gestion complète des procédures de recouvrement et d'expulsion en cas d'impayés, sans stress pour vous.",
                    icon: <FileText className="h-10 w-10 text-white" />,
                  },
                  {
                    title: "Protection complète",
                    description:
                      "Couverture des dégradations immobilières causées par le locataire et des frais de contentieux.",
                    icon: <Banknote className="h-10 w-10 text-white" />,
                  },
                ].map((advantage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-4">
                      {advantage.icon}
                    </div>
                    <h3 className="text-xl font-bold mb-2">{advantage.title}</h3>
                    <p className="text-gray-700">{advantage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section de couverture */}
            <section className="mb-12" id="couverture">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Ce que couvre notre Garantie Loyers Impayés</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    title: "Loyers impayés",
                    description:
                      "Prise en charge des loyers et charges impayés jusqu'à 24 mois, avec un plafond mensuel adapté à votre bien.",
                    included: true,
                  },
                  {
                    title: "Frais de procédure",
                    description:
                      "Couverture des frais d'huissier, d'avocat et de procédure d'expulsion en cas de contentieux avec le locataire.",
                    included: true,
                  },
                  {
                    title: "Dégradations locatives",
                    description:
                      "Indemnisation des dommages causés par le locataire au-delà du dépôt de garantie, jusqu'à 10 000€.",
                    included: true,
                  },
                  {
                    title: "Protection juridique",
                    description:
                      "Assistance juridique pour tous les litiges liés à la location de votre bien immobilier.",
                    included: true,
                  },
                  {
                    title: "Défense-recours",
                    description:
                      "Prise en charge des frais de défense de vos intérêts en cas de litige avec le locataire.",
                    included: true,
                  },
                  {
                    title: "Accompagnement personnalisé",
                    description:
                      "Suivi personnalisé de votre dossier par un conseiller dédié tout au long de la procédure.",
                    included: true,
                  },
                ].map((coverage, index) => (
                  <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                    <div className="flex items-center mb-3">
                      <div
                        className={`w-8 h-8 rounded-full flex items-center justify-center mr-3 ${
                          coverage.included ? "bg-green-100 text-green-600" : "bg-gray-100 text-gray-400"
                        }`}
                      >
                        {coverage.included ? <Check className="h-5 w-5" /> : "X"}
                      </div>
                      <h3 className="text-xl font-semibold">{coverage.title}</h3>
                    </div>
                    <p className="text-gray-700">{coverage.description}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section de fonctionnement */}
            <section className="mb-12" id="fonctionnement">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Comment fonctionne notre Garantie Loyers Impayés ?
              </h2>
              <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100 mb-8">
                <div className="space-y-8">
                  {[
                    {
                      step: "1",
                      title: "Sélection du locataire",
                      description:
                        "Nous vérifions la solvabilité du locataire selon les critères de l'assurance pour garantir l'éligibilité à la GLI.",
                    },
                    {
                      step: "2",
                      title: "Souscription de la garantie",
                      description:
                        "Une fois le locataire validé, nous souscrivons la garantie loyers impayés pour votre bien.",
                    },
                    {
                      step: "3",
                      title: "Suivi des paiements",
                      description:
                        "Nous assurons un suivi rigoureux des paiements des loyers et intervenons dès le premier retard.",
                    },
                    {
                      step: "4",
                      title: "Gestion des impayés",
                      description:
                        "En cas d'impayé, nous déclenchons les procédures de recouvrement et activons la garantie pour maintenir vos revenus.",
                    },
                    {
                      step: "5",
                      title: "Indemnisation",
                      description:
                        "Vous recevez les indemnisations pour les loyers impayés pendant toute la durée de la procédure, jusqu'à 24 mois.",
                    },
                  ].map((item) => (
                    <div key={item.step} className="flex">
                      <div className="mr-6">
                        <div className="w-10 h-10 rounded-full bg-[#002395] text-white flex items-center justify-center font-bold">
                          {item.step}
                        </div>
                      </div>
                      <div>
                        <h4 className="text-xl font-medium mb-2">{item.title}</h4>
                        <p className="text-gray-700">{item.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </section>

            {/* Section Pourquoi nous choisir */}
            <section className="mb-12" id="pourquoi-nous">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Pourquoi choisir DOGES pour votre Garantie Loyers Impayés ?
              </h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    title: "Expertise reconnue",
                    description:
                      "Plus de 15 ans d'expérience dans la gestion locative et l'assurance loyers impayés en Île-de-France.",
                    icon: <Shield className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Tarifs compétitifs",
                    description:
                      "Des tarifs négociés auprès de nos partenaires assureurs pour vous offrir le meilleur rapport qualité-prix.",
                    icon: <Banknote className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Accompagnement juridique",
                    description:
                      "Une équipe juridique dédiée pour vous accompagner dans toutes les démarches en cas de litige.",
                    icon: <Scale className="h-6 w-6 text-[#002395]" />,
                  },
                  {
                    title: "Réactivité",
                    description:
                      "Intervention rapide dès le premier impayé pour maximiser les chances de recouvrement amiable.",
                    icon: <Clock className="h-6 w-6 text-[#002395]" />,
                  },
                ].map((item, index) => (
                  <div key={index} className="flex p-4 bg-white rounded-lg shadow-sm border border-gray-100">
                    <div className="mr-4 mt-1">{item.icon}</div>
                    <div>
                      <h3 className="text-lg font-semibold mb-2">{item.title}</h3>
                      <p className="text-gray-700">{item.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Section FAQ */}
            <section className="mb-12" id="faq">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Questions fréquentes sur la Garantie Loyers Impayés
              </h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Quel est le coût d'une garantie loyers impayés ?",
                    answer:
                      "Le coût d'une garantie loyers impayés varie généralement entre 2,5% et 3,5% du montant annuel des loyers et charges. Ce tarif dépend de plusieurs facteurs comme le type de bien, sa localisation et le profil du locataire. Chez DOGES, nous négocions les meilleures conditions avec nos partenaires assureurs pour vous offrir un rapport qualité-prix optimal.",
                  },
                  {
                    question: "Quelles sont les conditions pour bénéficier de la garantie loyers impayés ?",
                    answer:
                      "Pour bénéficier de la garantie loyers impayés, le locataire doit généralement présenter un taux d'effort (ratio loyer/revenus) inférieur à 33%, être en CDI hors période d'essai ou avoir un statut stable (fonctionnaire, retraité), et ne pas avoir d'incidents de paiement récents. Notre équipe vérifie rigoureusement ces critères pour garantir l'éligibilité à la garantie.",
                  },
                  {
                    question: "La garantie loyers impayés est-elle obligatoire ?",
                    answer:
                      "Non, la garantie loyers impayés n'est pas obligatoire, mais elle est fortement recommandée pour tout propriétaire bailleur souhaitant sécuriser ses revenus locatifs. Elle offre une protection complète contre les risques d'impayés, les dégradations et les frais de procédure, vous permettant de profiter sereinement de votre investissement immobilier.",
                  },
                  {
                    question: "Quelle est la différence entre la GLI et Visale ?",
                    answer:
                      "La GLI (Garantie Loyers Impayés) est une assurance privée souscrite par le propriétaire, tandis que Visale est une garantie publique gratuite proposée par Action Logement. La GLI offre généralement une couverture plus étendue (jusqu'à 24 mois d'impayés, dégradations, frais juridiques) mais est payante, alors que Visale est gratuite mais avec des conditions d'éligibilité plus restrictives et une couverture limitée.",
                  },
                  {
                    question: "Comment sont gérés les impayés avec la GLI ?",
                    answer:
                      "En cas d'impayé, nous engageons immédiatement une procédure de relance auprès du locataire. Si l'impayé persiste, nous déclarons le sinistre à l'assureur qui prend en charge l'indemnisation des loyers impayés. Parallèlement, nous lançons les procédures juridiques nécessaires pour le recouvrement de la dette et, si nécessaire, l'expulsion du locataire. Vous continuez à percevoir vos loyers pendant toute la durée de la procédure.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
            </section>

            {/* Section Témoignages */}
            <section className="mb-12" id="temoignages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Ce que disent nos clients</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    name: "Sophie M.",
                    location: "Propriétaire à Paris 16ème",
                    testimonial:
                      "Après une mauvaise expérience avec un locataire, j'ai souscrit à la GLI avec DOGES. Quelques mois plus tard, mon nouveau locataire a perdu son emploi et a cessé de payer. DOGES a géré toute la procédure et je n'ai subi aucune perte financière. Un vrai soulagement !",
                    rating: 5,
                  },
                  {
                    name: "Jean-Pierre D.",
                    location: "Propriétaire à Neuilly-sur-Seine",
                    testimonial:
                      "La réactivité de l'équipe DOGES est impressionnante. Dès le premier retard de paiement, ils ont pris les choses en main. La garantie loyers impayés m'a permis de continuer à percevoir mes revenus locatifs sans interruption pendant toute la procédure.",
                    rating: 5,
                  },
                  {
                    name: "Marie L.",
                    location: "Propriétaire à Boulogne-Billancourt",
                    testimonial:
                      "J'hésitais à souscrire une GLI en raison du coût, mais après avoir fait le calcul, c'est un investissement très rentable pour la tranquillité d'esprit qu'elle procure. Le service client de DOGES est exemplaire et très professionnel.",
                    rating: 4,
                  },
                  {
                    name: "Thomas B.",
                    location: "Propriétaire à Levallois-Perret",
                    testimonial:
                      "Suite à des dégradations importantes dans mon appartement, la GLI a couvert tous les frais de remise en état au-delà du dépôt de garantie. DOGES a géré tout le dossier avec efficacité, me permettant de relouer rapidement mon bien.",
                    rating: 5,
                  },
                ].map((testimonial, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <div className="flex items-center mb-4">
                      {[...Array(testimonial.rating)].map((_, i) => (
                        <svg
                          key={i}
                          className="w-5 h-5 text-yellow-400"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                          xmlns="http://www.w3.org/2000/svg"
                        >
                          <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"></path>
                        </svg>
                      ))}
                    </div>
                    <p className="text-gray-700 mb-4 italic">"{testimonial.testimonial}"</p>
                    <div>
                      <p className="font-semibold">{testimonial.name}</p>
                      <p className="text-sm text-gray-500">{testimonial.location}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Formulaire de contact */}
            <section id="contact">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Demandez un devis pour votre Garantie Loyers Impayés
              </h2>
              <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                <SmartContactForm />
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="space-y-8">
            {/* Widget Table des matières */}
            <TableOfContentsWidget
              items={[
                { id: "presentation", title: "Présentation" },
                { id: "avantages", title: "Avantages" },
                { id: "couverture", title: "Couverture" },
                { id: "fonctionnement", title: "Fonctionnement" },
                { id: "pourquoi-nous", title: "Pourquoi nous choisir" },
                { id: "faq", title: "FAQ" },
                { id: "temoignages", title: "Témoignages" },
                { id: "contact", title: "Contact" },
              ]}
            />

            {/* Widget Contact */}
            <ContactWidget/>

            {/* Widget Services associés */}
            <RelatedServicesWidget
              title="Services associés"
              services={[
                {
                  title: "Gestion Locative",
                  description: "Confiez-nous la gestion complète de votre bien immobilier",
                  link: "/services/gestion-locative",
                },
                {
                  title: "Assurance PNO",
                  description: "Protégez votre bien immobilier avec notre assurance propriétaire non occupant",
                  link: "/services/assurance-pno",
                },
                {
                  title: "Service Juridique",
                  description: "Bénéficiez de notre expertise juridique pour sécuriser votre investissement",
                  link: "/services/juridique",
                },
              ]}
            />
          </div>
        </div>
      </div>
    </>
  )
}
