import type { Metadata } from "next"
import Link from "next/link"
import { villesData } from "./villes-data"
import FranceVilleCard from "@/components/france-ville-card"
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion"
// Add ServiceHeader import
import ServiceHeader from "@/components/service-header"

export const metadata: Metadata = {
  title: "Gestion Locative en France | Agence Immobilière DOGES",
  description:
    "DOGES, agence immobilière spécialisée en gestion locative clés en main dans toute la France. Confiez-nous la gestion de votre appartement ou maison : recherche de locataire, vérification des dossiers, gestion financière, technique et administrative.",
  keywords: [
    "gestion locative France",
    "agence immobilière",
    "gestion immobilière",
    "externalisation immobilière",
    "gestion locative clés en main",
    "location appartement",
    "location maison",
    "recherche locataire",
    "vérification dossiers locataires",
    "gestion financière immobilier",
    "gestion technique immobilier",
    "gestion administrative immobilier",
    "vente immobilière",
    "agences immobilières autour de moi",
    "commercialisation bien immobilier",
  ],
  openGraph: {
    title: "Gestion Locative en France | Agence Immobilière DOGES",
    description:
      "DOGES, agence immobilière spécialisée en gestion locative clés en main dans toute la France. Confiez-nous la gestion de votre appartement ou maison.",
    url: "https://www.dogesadb.fr/services/gestion-locative-france",
    siteName: "DOGES",
    locale: "fr_FR",
    type: "website",
  },
}

// Replace the custom header implementation with ServiceHeader at the beginning of the component
export default function GestionLocativeFrancePage() {
  return (
    <div className="bg-white">
      <ServiceHeader
        title="Gestion Locative en France"
        description="Services professionnels de gestion locative dans toute la France. Confiez la gestion de votre patrimoine immobilier à nos experts."
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Gestion Locative France", url: "/services/gestion-locative-france" },
        ]}
        backgroundImage="/paris-panorama.png"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-4xl mx-auto mb-12">
          <h2 className="text-3xl font-bold mb-6 text-[#002395]" id="presence">
            Notre présence nationale en tant qu'agence immobilière
          </h2>
          <p className="text-lg mb-4">
            DOGES est fier de vous proposer ses services de <strong>gestion immobilière</strong> dans les principales
            villes de France. Notre réseau d'experts locaux connaît parfaitement les spécificités de chaque marché
            immobilier, que ce soit pour la <strong>location d'appartements</strong>, de <strong>maisons</strong> ou de{" "}
            <strong>locaux commerciaux</strong>.
          </p>
          <p className="mb-6">
            Que vous possédiez un bien à Lyon, Marseille, Bordeaux ou dans toute autre grande ville française, nos
            équipes sont à votre disposition pour vous offrir un service de <strong>gestion locative sur mesure</strong>
            , adapté aux particularités locales. Nous sommes l'<strong>agence immobilière autour de vous</strong> qui
            comprend vos besoins spécifiques.
          </p>

          <div className="bg-blue-50 p-6 rounded-lg mb-8">
            <h3 className="text-xl font-semibold mb-3 text-[#002395]">
              Pourquoi choisir DOGES pour la gestion de votre bien ?
            </h3>
            <ul className="list-disc pl-5 space-y-2">
              <li>Une expertise locale dans chaque ville où nous sommes présents</li>
              <li>Des honoraires transparents et compétitifs</li>
              <li>Un suivi personnalisé de votre bien</li>
              <li>
                Une <strong>sélection rigoureuse des locataires</strong> et <strong>vérification des dossiers</strong>
              </li>
              <li>
                Une <strong>gestion financière, technique et administrative complète</strong>
              </li>
              <li>
                Des solutions d'<strong>externalisation immobilière</strong> pour les propriétaires et investisseurs
              </li>
            </ul>
          </div>

          <div id="services" className="mb-12">
            <h2 className="text-3xl font-bold mb-6 text-[#002395]">Nos services de gestion locative clés en main</h2>

            <div className="grid md:grid-cols-2 gap-6 mb-8">
              <div className="bg-white p-6 rounded-lg shadow-md">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">Commercialisation du bien</h3>
                <ul className="space-y-2">
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Estimation locative précise
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Reportage photos professionnel
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Diffusion sur les plateformes immobilières
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Organisation des visites
                  </li>
                </ul>
              </div>

              <div className="bg-white p-6 rounded-lg shadow-md">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">Recherche et sélection de locataires</h3>
                <ul className="space-y-2">
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Étude approfondie des dossiers
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Vérification de solvabilité
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Rédaction du bail conforme à la législation
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    État des lieux d'entrée détaillé
                  </li>
                </ul>
              </div>

              <div className="bg-white p-6 rounded-lg shadow-md">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">Gestion financière</h3>
                <ul className="space-y-2">
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Encaissement des loyers et charges
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Révision annuelle des loyers
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Régularisation des charges
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Gestion des impayés et relances
                  </li>
                </ul>
              </div>

              <div className="bg-white p-6 rounded-lg shadow-md">
                <h3 className="text-xl font-semibold mb-3 text-[#002395]">Gestion technique et administrative</h3>
                <ul className="space-y-2">
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Suivi des travaux et interventions
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Gestion des sinistres et assurances
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Déclarations fiscales et aide à l'optimisation
                  </li>
                  <li className="flex items-start">
                    <svg
                      className="h-5 w-5 text-green-500 mr-2 mt-0.5"
                      fill="none"
                      viewBox="0 0 24 24"
                      stroke="currentColor"
                    >
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                    </svg>
                    Veille juridique et mise en conformité
                  </li>
                </ul>
              </div>
            </div>

            <div className="bg-[#002395]/5 p-6 rounded-lg">
              <h3 className="text-xl font-semibold mb-3 text-[#002395]">Services complémentaires</h3>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="flex items-start">
                  <svg
                    className="h-5 w-5 text-[#002395] mr-2 mt-0.5"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth="2"
                      d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                    />
                  </svg>
                  <span>
                    <strong>Vente immobilière</strong> : estimation et commercialisation de votre bien
                  </span>
                </div>
                <div className="flex items-start">
                  <svg
                    className="h-5 w-5 text-[#002395] mr-2 mt-0.5"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth="2"
                      d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                    />
                  </svg>
                  <span>
                    <strong>Assurance loyers impayés</strong> pour sécuriser vos revenus
                  </span>
                </div>
                <div className="flex items-start">
                  <svg
                    className="h-5 w-5 text-[#002395] mr-2 mt-0.5"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth="2"
                      d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                    />
                  </svg>
                  <span>
                    Conseils en <strong>investissement immobilier</strong>
                  </span>
                </div>
                <div className="flex items-start">
                  <svg
                    className="h-5 w-5 text-[#002395] mr-2 mt-0.5"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth="2"
                      d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                    />
                  </svg>
                  <span>
                    <strong>Externalisation immobilière</strong> pour les propriétaires multi-biens
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>

        <h2 className="text-3xl font-bold mb-6 text-[#002395] text-center">
          Nos agences de gestion locative en France
        </h2>
        <p className="text-center max-w-3xl mx-auto mb-8">
          Découvrez nos <strong>agences immobilières</strong> spécialisées en <strong>gestion locative</strong> dans les
          principales villes de France. Cliquez sur une ville pour en savoir plus sur nos services de{" "}
          <strong>gestion de location d'appartements et maisons</strong> dans cette région.
        </p>

        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mb-12">
          {villesData.map((ville) => (
            <FranceVilleCard
              key={ville.slug}
              nom={ville.nom}
              slug={ville.slug}
              region={ville.region}
              description={ville.shortDescription || undefined}
              imagePath={getVilleImage(ville.nom)}
            />
          ))}
        </div>

        <div className="max-w-4xl mx-auto mt-12 mb-12">
          <h2 className="text-3xl font-bold mb-6 text-[#002395]">Questions fréquentes sur la gestion locative</h2>

          <Accordion type="single" collapsible className="w-full">
            <AccordionItem value="item-1">
              <AccordionTrigger className="text-left">
                Qu'est-ce que la gestion locative clés en main ?
              </AccordionTrigger>
              <AccordionContent>
                La <strong>gestion locative clés en main</strong> est un service complet qui prend en charge tous les
                aspects de la location de votre bien immobilier :<strong>recherche de locataire</strong>,{" "}
                <strong>vérification des dossiers</strong>, rédaction des baux, états des lieux,
                <strong>gestion financière</strong>, <strong>gestion technique</strong> et{" "}
                <strong>gestion administrative</strong>. Ce service vous permet de déléguer entièrement la gestion de
                votre bien à notre <strong>agence immobilière</strong>.
              </AccordionContent>
            </AccordionItem>

            <AccordionItem value="item-2">
              <AccordionTrigger className="text-left">
                Quels sont les avantages de faire appel à une agence pour la gestion locative ?
              </AccordionTrigger>
              <AccordionContent>
                Faire appel à une <strong>agence immobilière</strong> pour la <strong>gestion locative</strong> présente
                de nombreux avantages :
                <ul className="list-disc pl-5 mt-2 space-y-1">
                  <li>Gain de temps considérable</li>
                  <li>Expertise juridique et fiscale</li>
                  <li>Sélection rigoureuse des locataires réduisant les risques d'impayés</li>
                  <li>Gestion professionnelle des problèmes techniques</li>
                  <li>Optimisation de votre rentabilité</li>
                  <li>Tranquillité d'esprit</li>
                </ul>
              </AccordionContent>
            </AccordionItem>

            <AccordionItem value="item-3">
              <AccordionTrigger className="text-left">
                Comment trouver une agence immobilière de gestion locative près de chez moi ?
              </AccordionTrigger>
              <AccordionContent>
                Pour trouver une <strong>agence immobilière autour de vous</strong> spécialisée en{" "}
                <strong>gestion locative</strong>, vous pouvez consulter notre réseau d'agences présentes dans les
                principales villes de France. DOGES dispose d'experts locaux dans chaque région qui connaissent
                parfaitement les spécificités du marché immobilier local. N'hésitez pas à nous contacter pour être mis
                en relation avec l'agence la plus proche de votre bien.
              </AccordionContent>
            </AccordionItem>

            <AccordionItem value="item-4">
              <AccordionTrigger className="text-left">Quels types de biens gérez-vous ?</AccordionTrigger>
              <AccordionContent>
                Notre <strong>agence de gestion de location</strong> prend en charge tous types de biens immobiliers :
                <ul className="list-disc pl-5 mt-2 space-y-1">
                  <li>
                    <strong>Appartements</strong> de toutes tailles (studios, T1, T2, T3, etc.)
                  </li>
                  <li>
                    <strong>Maisons</strong> individuelles ou mitoyennes
                  </li>
                  <li>Locaux commerciaux</li>
                  <li>Bureaux</li>
                  <li>Immeubles entiers</li>
                </ul>
                Que vous possédiez un studio ou un portefeuille immobilier complet, nos services s'adaptent à vos
                besoins.
              </AccordionContent>
            </AccordionItem>

            <AccordionItem value="item-5">
              <AccordionTrigger className="text-left">Qu'est-ce que l'externalisation immobilière ?</AccordionTrigger>
              <AccordionContent>
                L'<strong>externalisation immobilière</strong> est un service destiné aux propriétaires de plusieurs
                biens ou aux investisseurs qui souhaitent déléguer l'ensemble de la gestion de leur patrimoine
                immobilier. Ce service va au-delà de la simple
                <strong>gestion locative</strong> en incluant des conseils stratégiques sur l'optimisation de votre
                patrimoine, la planification des travaux, l'analyse de la rentabilité et même l'accompagnement pour de
                nouveaux investissements.
              </AccordionContent>
            </AccordionItem>

            <AccordionItem value="item-6">
              <AccordionTrigger className="text-left">
                Proposez-vous aussi des services de vente immobilière ?
              </AccordionTrigger>
              <AccordionContent>
                Oui, en plus de nos services de <strong>gestion locative</strong>, nous proposons également des services
                de
                <strong>vente immobilière</strong>. Notre équipe d'experts peut vous accompagner dans l'estimation de
                votre bien, sa mise en valeur, sa commercialisation et toutes les démarches jusqu'à la signature chez le
                notaire. Nous pouvons également vous conseiller sur le meilleur moment pour vendre votre bien locatif.
              </AccordionContent>
            </AccordionItem>
          </Accordion>
        </div>

        <div className="max-w-4xl mx-auto bg-gray-50 p-8 rounded-lg">
          <h2 className="text-2xl font-bold mb-6 text-[#002395]">Vous ne trouvez pas votre ville ?</h2>
          <p className="mb-6">
            Même si votre ville n'est pas listée ci-dessus, il est possible que nous y proposions nos services de
            <strong>gestion locative</strong> et d'<strong>agence immobilière</strong>. Contactez-nous pour discuter de
            votre projet immobilier, où qu'il se trouve en France.
          </p>
          <Link
            href="/contact"
            className="inline-block bg-[#002395] hover:bg-[#001a70] text-white font-medium py-3 px-6 rounded-md transition-colors"
          >
            Nous contacter
          </Link>
        </div>
      </div>

      {/* Données structurées pour le SEO */}
      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{
          __html: JSON.stringify({
            "@context": "https://schema.org",
            "@type": "RealEstateAgent",
            name: "DOGES",
            description: "Agence immobilière spécialisée en gestion locative clés en main dans toute la France",
            url: "https://www.dogesadb.fr/services/gestion-locative-france",
            logo: "https://www.dogesadb.fr/logo.png",
            address: {
              "@type": "PostalAddress",
              addressCountry: "FR",
            },
            telephone: "+33123456789",
            email: "contact@dogesadb.fr",
            areaServed: [
              {
                "@type": "City",
                name: "Paris",
              },
              {
                "@type": "City",
                name: "Lyon",
              },
              {
                "@type": "City",
                name: "Marseille",
              },
              {
                "@type": "City",
                name: "Bordeaux",
              },
              {
                "@type": "City",
                name: "Nice",
              },
            ],
            hasOfferCatalog: {
              "@type": "OfferCatalog",
              name: "Services de gestion immobilière",
              itemListElement: [
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Gestion Locative Clés en Main",
                    description: "Service complet de gestion locative pour propriétaires",
                  },
                },
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Recherche de Locataires",
                    description: "Sélection rigoureuse et vérification des dossiers",
                  },
                },
                {
                  "@type": "Offer",
                  itemOffered: {
                    "@type": "Service",
                    name: "Gestion Financière et Administrative",
                    description: "Gestion complète des aspects financiers et administratifs",
                  },
                },
              ],
            },
          }),
        }}
      />
    </div>
  )
}

// Fonction pour obtenir l'image de la ville
function getVilleImage(ville: string) {
  const villeSlug = ville.toLowerCase()

  // Utiliser les images existantes pour les villes principales
  if (villeSlug === "nice") {
    return "/nice-promenade-des-anglais.png"
  } else if (villeSlug === "lyon") {
    return "/lyon-france-skyline.png"
  } else if (villeSlug === "marseille") {
    return "/marseille-port.png"
  } else if (villeSlug === "toulouse") {
    return "/toulouse-pink-city.png"
  } else if (villeSlug === "nantes") {
    return "/nantes-france-city.png"
  } else if (villeSlug === "lille") {
    return "/lille-grand-place.png"
  } else if (villeSlug === "montpellier") {
    return "/montpellier-place-comedie.png"
  } else if (villeSlug === "strasbourg") {
    return "/strasbourg-cathedral.png"
  } else if (villeSlug === "aix-en-provence") {
    return "/aix-en-provence-fontaine.png"
  } else if (villeSlug === "bordeaux") {
    return "/bordeaux-place-bourse.png"
  } else {
    // Image par défaut si la ville n'est pas dans notre liste
    return "/immobilier-scene.png"
  }
}
