import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check, ArrowRight, MapPin, Building, Users, TrendingUp } from "lucide-react"
import { Card, CardContent } from "@/components/ui/card"
import ServiceHeader from "@/components/service-header"
import ContactWidget from "@/components/contact-widget"
import RelatedServicesWidget from "@/components/related-services-widget"
import TableOfContentsWidget from "@/components/table-of-contents-widget"
import { ConsultationCTA } from "@/components/consultation-cta"

export const metadata: Metadata = {
  title: "Gestion Locative Paris | DOGES Administrateur de Biens",
  description:
    "Service de gestion locative spécialisé à Paris. Expertise du marché parisien, connaissance des arrondissements et gestion adaptée aux spécificités de la capitale.",
  keywords: "gestion locative Paris, administrateur biens Paris, gestion appartement Paris, location Paris",
}

export default function GestionLocativeParisPage() {
  // Configuration de la table des matières
  const tocItems = [
    { id: "presentation", title: "Gestion locative à Paris" },
    { id: "specificites", title: "Spécificités du marché parisien" },
    { id: "arrondissements", title: "Expertise par arrondissement" },
    { id: "services", title: "Nos services spécialisés" },
    { id: "avantages", title: "Pourquoi nous choisir" },
    { id: "temoignages", title: "Témoignages clients" },
    { id: "faq", title: "Questions fréquentes" },
  ]

  // Services associés pertinents
  const relatedServices = [
    {
      title: "Gestion Locative Île-de-France",
      description: "Nos services étendus à toute la région francilienne",
      link: "/services/gestion-locative-ile-de-france",
    },
    {
      title: "Administrateur de Biens Paris",
      description: "Administration complète de votre patrimoine parisien",
      link: "/services/administrateur-biens-paris",
    },
    {
      title: "Transaction Immobilière",
      description: "Achat et vente de biens immobiliers à Paris",
      link: "/services/transaction",
    },
  ]

  return (
    <>
      {/* En-tête de service avec image de fond */}
      <ServiceHeader
        title="Gestion Locative Paris"
        description="Une expertise locale pour optimiser la gestion de vos biens immobiliers dans la capitale"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Gestion Locative Paris", url: "/services/gestion-locative-paris" },
        ]}
        backgroundImage="/images/paris-real-estate.png"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">
            {/* Présentation */}
            <section className="mb-16" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Gestion locative spécialisée à Paris</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    Paris représente un marché immobilier unique avec ses propres règles, contraintes et opportunités.
                    Chez DOGES, nous maîtrisons parfaitement les spécificités de chaque arrondissement parisien pour
                    optimiser la gestion de votre patrimoine immobilier.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Notre expertise locale nous permet de naviguer efficacement dans la complexité du marché parisien :
                    encadrement des loyers, réglementations spécifiques, typologie des locataires et dynamiques de
                    quartier.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Connaissance approfondie de tous les arrondissements",
                      "Maîtrise de l'encadrement des loyers parisiens",
                      "Réseau local de prestataires qualifiés",
                      "Expertise des copropriétés parisiennes",
                      "Suivi personnalisé adapté au marché local",
                    ].map((feature, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{feature}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/images/paris-real-estate.png"
                    alt="Gestion locative Paris - Vue sur immeubles parisiens"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Spécificités du marché parisien */}
            <section className="mb-16 bg-gray-50 -mx-4 px-4 py-12" id="specificites">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Spécificités du marché locatif parisien
              </h2>

              <div className="grid md:grid-cols-2 gap-8">
                <Card className="bg-white rounded-xl shadow-lg border border-gray-100">
                  <CardContent className="p-6">
                    <div className="flex items-center mb-4">
                      <MapPin className="h-8 w-8 text-[#002395] mr-3" />
                      <h3 className="text-xl font-bold">Encadrement des loyers</h3>
                    </div>
                    <p className="text-gray-700 mb-4">
                      Paris applique un encadrement strict des loyers avec des plafonds par zone et par type de
                      logement. Notre expertise nous permet de fixer le loyer optimal dans le respect de la
                      réglementation.
                    </p>
                    <ul className="space-y-2">
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Connaissance des loyers de référence par secteur</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Optimisation dans le respect des plafonds</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Veille réglementaire permanente</span>
                      </li>
                    </ul>
                  </CardContent>
                </Card>

                <Card className="bg-white rounded-xl shadow-lg border border-gray-100">
                  <CardContent className="p-6">
                    <div className="flex items-center mb-4">
                      <Building className="h-8 w-8 text-[#002395] mr-3" />
                      <h3 className="text-xl font-bold">Copropriétés parisiennes</h3>
                    </div>
                    <p className="text-gray-700 mb-4">
                      Les immeubles parisiens, souvent anciens et classés, nécessitent une gestion technique
                      spécialisée. Nous maîtrisons les contraintes architecturales et réglementaires spécifiques.
                    </p>
                    <ul className="space-y-2">
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Expertise des immeubles haussmanniens</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Gestion des contraintes patrimoniales</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Réseau d'artisans spécialisés</span>
                      </li>
                    </ul>
                  </CardContent>
                </Card>

                <Card className="bg-white rounded-xl shadow-lg border border-gray-100">
                  <CardContent className="p-6">
                    <div className="flex items-center mb-4">
                      <Users className="h-8 w-8 text-[#002395] mr-3" />
                      <h3 className="text-xl font-bold">Profils de locataires</h3>
                    </div>
                    <p className="text-gray-700 mb-4">
                      Le marché parisien attire une clientèle diversifiée : jeunes actifs, expatriés, étudiants,
                      familles. Nous adaptons notre approche selon le profil recherché pour votre bien.
                    </p>
                    <ul className="space-y-2">
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Sélection adaptée au type de bien</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Accompagnement des expatriés</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Vérification renforcée des dossiers</span>
                      </li>
                    </ul>
                  </CardContent>
                </Card>

                <Card className="bg-white rounded-xl shadow-lg border border-gray-100">
                  <CardContent className="p-6">
                    <div className="flex items-center mb-4">
                      <TrendingUp className="h-8 w-8 text-[#002395] mr-3" />
                      <h3 className="text-xl font-bold">Dynamique du marché</h3>
                    </div>
                    <p className="text-gray-700 mb-4">
                      Paris connaît une forte demande locative avec des variations selon les quartiers et les saisons.
                      Notre connaissance fine du marché nous permet d'optimiser vos revenus locatifs.
                    </p>
                    <ul className="space-y-2">
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Analyse des tendances par arrondissement</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Stratégie de commercialisation adaptée</span>
                      </li>
                      <li className="flex items-start">
                        <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                        <span>Optimisation des délais de location</span>
                      </li>
                    </ul>
                  </CardContent>
                </Card>
              </div>
            </section>

            {/* Expertise par arrondissement */}
            <section className="mb-16" id="arrondissements">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Notre expertise par arrondissement</h2>
              <p className="text-lg text-gray-700 mb-8">
                Chaque arrondissement parisien a ses propres caractéristiques. Notre équipe connaît parfaitement les
                spécificités de chaque quartier pour optimiser la gestion de votre bien.
              </p>

              <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4">
                {[
                  { arr: "1er - 4e", desc: "Centre historique, clientèle haut de gamme" },
                  { arr: "5e - 6e", desc: "Quartier Latin, étudiants et intellectuels" },
                  { arr: "7e - 8e", desc: "Prestige et affaires, expatriés" },
                  { arr: "9e - 10e", desc: "Dynamique et branché, jeunes actifs" },
                  { arr: "11e - 12e", desc: "Familial et accessible, diversité" },
                  { arr: "13e - 14e", desc: "Moderne et international" },
                  { arr: "15e - 16e", desc: "Résidentiel et familial" },
                  { arr: "17e - 20e", desc: "Émergent et créatif" },
                ].map((zone, index) => (
                  <div key={index} className="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                    <h4 className="font-semibold text-[#002395] mb-2">{zone.arr}</h4>
                    <p className="text-sm text-gray-600">{zone.desc}</p>
                  </div>
                ))}
              </div>

              <div className="mt-8 bg-blue-50 p-6 rounded-lg border border-blue-100">
                <h3 className="text-xl font-semibold mb-3">Services spécialisés par arrondissement</h3>
                <div className="grid md:grid-cols-2 gap-4">
                  <Link
                    href="/services/gestion-locative-paris/1-2-3-4"
                    className="text-[#002395] hover:underline flex items-center"
                  >
                    <ArrowRight className="h-4 w-4 mr-2" />
                    Gestion locative Paris Centre (1er-4e)
                  </Link>
                  <Link
                    href="/services/gestion-locative-paris/5-6-7"
                    className="text-[#002395] hover:underline flex items-center"
                  >
                    <ArrowRight className="h-4 w-4 mr-2" />
                    Gestion locative Rive Gauche (5e-7e)
                  </Link>
                  <Link
                    href="/services/gestion-locative-paris/8-16-17"
                    className="text-[#002395] hover:underline flex items-center"
                  >
                    <ArrowRight className="h-4 w-4 mr-2" />
                    Gestion locative Ouest parisien (8e, 16e, 17e)
                  </Link>
                  <Link
                    href="/services/gestion-locative-paris/9-10-18"
                    className="text-[#002395] hover:underline flex items-center"
                  >
                    <ArrowRight className="h-4 w-4 mr-2" />
                    Gestion locative Nord-Est (9e, 10e, 18e)
                  </Link>
                </div>
              </div>
            </section>

            {/* Services spécialisés */}
            <section className="mb-16" id="services">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Nos services spécialisés pour Paris</h2>
              <div className="space-y-6">
                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Gestion de l'encadrement des loyers</h3>
                  <p className="text-gray-700 mb-4">
                    Nous maîtrisons parfaitement la réglementation parisienne sur l'encadrement des loyers pour
                    optimiser vos revenus dans le respect de la loi.
                  </p>
                  <ul className="grid md:grid-cols-2 gap-2">
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Calcul du loyer de référence majoré</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Justification des compléments de loyer</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Veille réglementaire continue</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Optimisation légale des revenus</span>
                    </li>
                  </ul>
                </div>

                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Gestion technique spécialisée</h3>
                  <p className="text-gray-700 mb-4">
                    Les immeubles parisiens nécessitent une expertise technique particulière que nous maîtrisons
                    parfaitement.
                  </p>
                  <ul className="grid md:grid-cols-2 gap-2">
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Expertise immeubles haussmanniens</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Gestion des contraintes patrimoniales</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Réseau d'artisans agréés</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Respect des normes parisiennes</span>
                    </li>
                  </ul>
                </div>

                <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-100">
                  <h3 className="text-xl font-semibold mb-3">Commercialisation optimisée</h3>
                  <p className="text-gray-700 mb-4">
                    Notre stratégie de commercialisation est adaptée aux spécificités du marché parisien pour réduire la
                    vacance locative.
                  </p>
                  <ul className="grid md:grid-cols-2 gap-2">
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Diffusion ciblée par quartier</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Réseau de partenaires locaux</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Marketing digital spécialisé</span>
                    </li>
                    <li className="flex items-start">
                      <Check className="h-4 w-4 text-green-600 mr-2 mt-1 flex-shrink-0" />
                      <span>Accompagnement expatriés</span>
                    </li>
                  </ul>
                </div>
              </div>
            </section>

            {/* Avantages */}
            <section className="mb-16" id="avantages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Pourquoi choisir DOGES pour Paris</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div className="order-2 lg:order-1">
                  <div className="space-y-6">
                    {[
                      {
                        title: "Expertise locale inégalée",
                        description:
                          "Plus de 15 ans d'expérience sur le marché parisien avec une connaissance fine de chaque arrondissement.",
                      },
                      {
                        title: "Maîtrise réglementaire",
                        description:
                          "Parfaite connaissance de l'encadrement des loyers et des spécificités juridiques parisiennes.",
                      },
                      {
                        title: "Réseau local privilégié",
                        description:
                          "Partenaires de confiance dans tous les arrondissements : artisans, syndics, notaires.",
                      },
                      {
                        title: "Réactivité parisienne",
                        description:
                          "Équipe basée à Paris pour une intervention rapide et une connaissance terrain permanente.",
                      },
                      {
                        title: "Clientèle diversifiée",
                        description:
                          "Base de locataires qualifiés adaptée à chaque type de bien et chaque arrondissement.",
                      },
                    ].map((advantage, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <div>
                          <h3 className="font-semibold">{advantage.title}</h3>
                          <p className="text-gray-700">{advantage.description}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="order-1 lg:order-2 relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/images/paris-real-estate.png"
                    alt="Équipe DOGES spécialisée Paris"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Témoignages */}
            <section className="mb-16" id="temoignages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Témoignages de propriétaires parisiens</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    name: "Marie-Claire D.",
                    location: "Propriétaire dans le 7e arrondissement",
                    testimonial:
                      "DOGES gère mon appartement près des Invalides depuis 3 ans. Leur connaissance du marché parisien et de l'encadrement des loyers m'a permis d'optimiser mes revenus tout en respectant la réglementation. Service impeccable !",
                  },
                  {
                    name: "Jean-Philippe M.",
                    location: "Investisseur multi-arrondissements",
                    testimonial:
                      "J'ai des biens dans plusieurs arrondissements parisiens. L'équipe DOGES adapte parfaitement sa stratégie selon chaque quartier. Leur expertise locale fait vraiment la différence pour trouver les bons locataires.",
                  },
                  {
                    name: "Sylvie L.",
                    location: "Propriétaire dans le Marais (4e)",
                    testimonial:
                      "Mon appartement dans un immeuble classé nécessitait une gestion technique spécialisée. DOGES maîtrise parfaitement les contraintes patrimoniales parisiennes. Très professionnels et réactifs.",
                  },
                  {
                    name: "Antoine B.",
                    location: "Propriétaire à Montparnasse (14e)",
                    testimonial:
                      "Grâce à DOGES, j'ai pu louer mon studio à un prix optimal dans le respect de l'encadrement des loyers. Leur expertise de la réglementation parisienne est un vrai plus pour maximiser la rentabilité.",
                  },
                ].map((testimonial, index) => (
                  <div key={index} className="bg-white p-6 rounded-xl shadow-sm border border-gray-100">
                    <div className="flex items-center mb-4">
                      <div className="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center text-gray-500 font-bold">
                        {testimonial.name.charAt(0)}
                      </div>
                      <div className="ml-4">
                        <h4 className="font-semibold">{testimonial.name}</h4>
                        <p className="text-sm text-gray-600">{testimonial.location}</p>
                      </div>
                    </div>
                    <p className="italic text-gray-700">"{testimonial.testimonial}"</p>
                  </div>
                ))}
              </div>
            </section>

            {/* FAQ */}
            <section className="mb-16" id="faq">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Questions fréquentes sur Paris</h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Comment fonctionne l'encadrement des loyers à Paris ?",
                    answer:
                      "L'encadrement des loyers à Paris fixe un loyer de référence et un loyer de référence majoré par secteur et par type de logement. Le loyer ne peut pas dépasser le loyer de référence majoré, sauf exceptions justifiées (caractéristiques exceptionnelles du logement). Nous maîtrisons parfaitement cette réglementation pour optimiser vos revenus dans le respect de la loi.",
                  },
                  {
                    question: "Quels sont les arrondissements les plus demandés pour la location ?",
                    answer:
                      "Les arrondissements centraux (1er-7e) et l'ouest parisien (8e, 16e, 17e) restent très demandés, notamment par les expatriés et cadres supérieurs. Les arrondissements de l'est (9e-11e, 20e) attirent une clientèle plus jeune et créative. Chaque secteur a ses spécificités que nous connaissons parfaitement pour adapter notre stratégie de commercialisation.",
                  },
                  {
                    question: "Comment gérez-vous les spécificités des immeubles haussmanniens ?",
                    answer:
                      "Les immeubles haussmanniens nécessitent une expertise particulière : contraintes architecturales, réglementations patrimoniales, techniques de rénovation spécifiques. Notre équipe technique connaît parfaitement ces enjeux et dispose d'un réseau d'artisans spécialisés dans ce type de bâtiments pour assurer un entretien optimal.",
                  },
                  {
                    question: "Quels sont vos délais moyens de location à Paris ?",
                    answer:
                      "À Paris, grâce à notre expertise locale et notre réseau, nous parvenons généralement à louer un bien en 2-4 semaines selon l'arrondissement et le type de bien. Notre connaissance fine du marché parisien et notre base de candidats qualifiés nous permettent d'optimiser les délais de commercialisation.",
                  },
                  {
                    question: "Accompagnez-vous les propriétaires expatriés ?",
                    answer:
                      "Oui, nous avons une expertise particulière dans l'accompagnement des propriétaires expatriés. Nous proposons des services adaptés : gestion à distance, reporting international, accompagnement fiscal, et nous maîtrisons les spécificités liées à la location à des locataires expatriés (garanties, dossiers internationaux, etc.).",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
              <div className="mt-6 text-center">
                <Link href="/faq" className="text-[#002395] font-medium hover:underline inline-flex items-center">
                  Voir toutes les questions fréquentes
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Link>
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="lg:w-1/3">
            {/* Table des matières */}
            <div className="mb-6">
              <TableOfContentsWidget items={tocItems} />
            </div>

            {/* Widget de contact rapide */}
            <ContactWidget />

            {/* Services associés */}
            <RelatedServicesWidget services={relatedServices} />
          </div>
        </div>

        {/* CTA Section */}
        <div className="mt-12">
          <ConsultationCTA
            title="Besoin d'une gestion locative spécialisée à Paris ?"
            description="Notre équipe d'experts parisiens vous accompagne pour optimiser la gestion de votre patrimoine immobilier dans la capitale."
            buttonText="Prendre rendez-vous"
            buttonLink="/contact"
          />
        </div>
      </div>
    </>
  )
}
