import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check, ArrowRight } from "lucide-react"
import { Card, CardContent } from "@/components/ui/card"
import TransactionCalculator from "./TransactionCalculator"
import SidebarCalculator from "./SidebarCalculator"
import ServiceHeader from "@/components/service-header"
import ContactWidget from "@/components/contact-widget"
import RelatedServicesWidget from "@/components/related-services-widget"
// 1. Importer le nouveau composant
import TableOfContentsWidget from "@/components/table-of-contents-widget"

export const metadata: Metadata = {
  title: "Transaction Immobilière à Paris | DOGES Immobilier",
  description:
    "Services de transaction immobilière à Paris. Achat, vente et estimation de biens immobiliers avec DOGES, votre partenaire de confiance.",
  keywords: "transaction immobilière Paris, achat immobilier Paris, vente immobilier Paris, estimation bien immobilier",
}

export default function TransactionPage() {
  // Configuration de la table des matières
  const tocItems = [
    { id: "presentation", title: "Notre expertise en transaction immobilière" },
    { id: "services", title: "Nos services de transaction" },
    { id: "avantages", title: "Pourquoi nous choisir" },
    { id: "calculateur", title: "Calculateur de transaction" },
    { id: "processus", title: "Notre processus" },
    { id: "temoignages", title: "Témoignages clients" },
    { id: "faq", title: "Questions fréquentes" },
  ]

  // Services associés pertinents
  const relatedServices = [
    {
      title: "Gestion Locative",
      description: "Confiez-nous la gestion complète de votre bien immobilier",
      link: "/services/gestion-locative",
    },
    {
      title: "Assurance Loyers Impayés",
      description: "Protégez vos revenus locatifs contre les impayés",
      link: "/services/assurance-loyers-impayes",
    },
    {
      title: "Gestion Immobilière Professionnelle",
      description: "Solutions complètes pour les propriétaires et investisseurs",
      link: "/services/gestion-immobiliere-professionnelle",
    },
  ]

  return (
    <>
      {/* En-tête de service avec image de fond */}
      <ServiceHeader
        title="Transaction Immobilière"
        description="Achat, vente et estimation de biens immobiliers à Paris et en Île-de-France"
        breadcrumbItems={[
          { name: "Accueil", url: "/" },
          { name: "Services", url: "/services" },
          { name: "Transaction Immobilière", url: "/services/transaction" },
        ]}
        backgroundImage="/agent-immobilier-transaction.png"
      />

      <div className="container mx-auto px-4 py-12">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Contenu principal */}
          <div className="lg:w-2/3">
            {/* Présentation */}
            <section className="mb-16" id="presentation">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Notre expertise en transaction immobilière</h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div>
                  <p className="text-lg text-gray-700 mb-6">
                    Chez DOGES, nous vous accompagnons dans toutes vos démarches d'achat et de vente immobilière à Paris
                    et en Île-de-France. Notre équipe d'experts connaît parfaitement le marché local et vous offre un
                    service personnalisé pour concrétiser votre projet dans les meilleures conditions.
                  </p>
                  <p className="text-lg text-gray-700 mb-6">
                    Que vous soyez à la recherche d'un bien d'exception, d'un investissement locatif ou que vous
                    souhaitiez vendre votre propriété, notre agence vous propose un accompagnement sur mesure et des
                    solutions adaptées à vos besoins.
                  </p>
                  <div className="space-y-4">
                    {[
                      "Estimation précise de votre bien",
                      "Stratégie de commercialisation personnalisée",
                      "Sélection rigoureuse des acquéreurs",
                      "Négociation optimale des conditions",
                      "Accompagnement jusqu'à la signature définitive",
                    ].map((feature, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <p>{feature}</p>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/images/transaction-service.webp"
                    alt="Agent immobilier présentant un bien à un couple"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Nos services de transaction */}
            <section className="mb-16 bg-gray-50 -mx-4 px-4 py-12" id="services">
              <h2 className="text-3xl font-bold mb-12 text-center text-[#002395]">
                Nos services de transaction immobilière
              </h2>

              {/* Tableau des services de transaction */}
              <div className="overflow-x-auto mb-8">
                <table className="w-full bg-white rounded-xl shadow-sm border-collapse">
                  <thead>
                    <tr className="bg-[#002395] text-white">
                      <th className="py-4 px-6 text-left rounded-tl-xl">Service</th>
                      <th className="py-4 px-6 text-left">Description</th>
                      <th className="py-4 px-6 text-left rounded-tr-xl">Avantages</th>
                    </tr>
                  </thead>
                  <tbody>
                    <tr className="border-b border-gray-200">
                      <td className="py-4 px-6 font-medium">Vente immobilière</td>
                      <td className="py-4 px-6">
                        Estimation, valorisation et commercialisation de votre bien avec une stratégie marketing sur
                        mesure.
                      </td>
                      <td className="py-4 px-6">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Estimation précise basée sur le marché local</li>
                          <li>Photos professionnelles et home staging virtuel</li>
                          <li>Diffusion sur les plateformes premium</li>
                        </ul>
                      </td>
                    </tr>
                    <tr className="border-b border-gray-200">
                      <td className="py-4 px-6 font-medium">Achat immobilier</td>
                      <td className="py-4 px-6">
                        Recherche personnalisée selon vos critères, visites sélectionnées et négociation pour acquérir
                        le bien idéal.
                      </td>
                      <td className="py-4 px-6">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Accès à des biens hors marché</li>
                          <li>Présélection rigoureuse pour gagner du temps</li>
                          <li>Accompagnement pour les visites et la négociation</li>
                        </ul>
                      </td>
                    </tr>
                    <tr>
                      <td className="py-4 px-6 font-medium rounded-bl-xl">Estimation immobilière</td>
                      <td className="py-4 px-6">
                        Évaluation précise de votre bien basée sur notre expertise du marché local et l'analyse
                        comparative.
                      </td>
                      <td className="py-4 px-6 rounded-br-xl">
                        <ul className="list-disc pl-5 space-y-1">
                          <li>Analyse détaillée du marché par quartier</li>
                          <li>Prise en compte des spécificités de votre bien</li>
                          <li>Rapport d'estimation complet offert</li>
                        </ul>
                      </td>
                    </tr>
                  </tbody>
                </table>
              </div>

              {/* Cartes de services avec plus de détails */}
              <div className="grid md:grid-cols-3 gap-8">
                {[
                  {
                    title: "Vente immobilière",
                    description:
                      "Notre service de vente immobilière comprend une stratégie personnalisée pour valoriser votre bien et attirer les meilleurs acquéreurs.",
                    details: [
                      "Estimation précise de votre bien",
                      "Reportage photos et vidéos professionnelles",
                      "Diffusion sur les plateformes premium",
                      "Organisation des visites avec des acquéreurs qualifiés",
                      "Négociation et sécurisation de la vente",
                    ],
                  },
                  {
                    title: "Achat immobilier",
                    description:
                      "Notre service d'achat immobilier vous fait gagner du temps en ciblant uniquement les biens correspondant à vos critères.",
                    details: [
                      "Définition précise de votre projet",
                      "Recherche active sur le marché et hors marché",
                      "Présélection rigoureuse des biens",
                      "Accompagnement lors des visites",
                      "Négociation et sécurisation de l'achat",
                    ],
                  },
                  {
                    title: "Estimation immobilière",
                    description:
                      "Notre service d'estimation immobilière vous fournit une évaluation précise de votre bien basée sur notre expertise du marché local.",
                    details: [
                      "Visite détaillée de votre bien",
                      "Analyse comparative du marché",
                      "Évaluation des points forts et des axes d'amélioration",
                      "Rapport d'estimation complet",
                      "Conseils pour optimiser la valeur de votre bien",
                    ],
                  },
                ].map((service, index) => (
                  <Card key={index} className="bg-white rounded-xl shadow-lg border border-gray-100">
                    <CardContent className="p-6">
                      <h3 className="text-xl font-bold mb-4">{service.title}</h3>
                      <p className="text-gray-700 mb-4">{service.description}</p>
                      <div className="space-y-2">
                        {service.details.map((detail, idx) => (
                          <div key={idx} className="flex items-start">
                            <div className="mt-1 mr-2 bg-[#002395] rounded-full p-0.5">
                              <Check className="h-3 w-3 text-white" />
                            </div>
                            <p className="text-sm">{detail}</p>
                          </div>
                        ))}
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>

            {/* Pourquoi nous choisir */}
            <section className="mb-16" id="avantages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Pourquoi nous choisir pour votre transaction immobilière
              </h2>
              <div className="grid lg:grid-cols-2 gap-12 items-center">
                <div className="order-2 lg:order-1">
                  <div className="space-y-6">
                    {[
                      {
                        title: "Expertise locale",
                        description:
                          "Une connaissance approfondie du marché immobilier parisien et de ses spécificités par quartier.",
                      },
                      {
                        title: "Réseau privilégié",
                        description:
                          "Un portefeuille de clients qualifiés et un réseau de partenaires pour faciliter vos transactions.",
                      },
                      {
                        title: "Transparence totale",
                        description: "Une communication claire et régulière tout au long du processus de transaction.",
                      },
                      {
                        title: "Négociation optimale",
                        description:
                          "Des techniques de négociation éprouvées pour obtenir les meilleures conditions pour nos clients.",
                      },
                      {
                        title: "Accompagnement juridique",
                        description:
                          "Un suivi rigoureux des aspects légaux et administratifs pour sécuriser votre transaction.",
                      },
                    ].map((advantage, index) => (
                      <div key={index} className="flex items-start">
                        <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                          <Check className="h-4 w-4 text-white" />
                        </div>
                        <div>
                          <h3 className="font-semibold">{advantage.title}</h3>
                          <p className="text-gray-700">{advantage.description}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
                <div className="order-1 lg:order-2 relative h-[400px] rounded-2xl overflow-hidden">
                  <Image
                    src="/images/transaction-service.webp"
                    alt="Conseiller immobilier professionnel"
                    fill
                    className="object-cover"
                  />
                </div>
              </div>
            </section>

            {/* Calculateur de transaction */}
            <section className="mb-16" id="calculateur">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Calculateur de transaction immobilière</h2>
              <p className="text-lg text-gray-700 mb-6">
                Utilisez notre calculateur pour estimer les frais liés à votre transaction immobilière, que ce soit pour
                un achat ou une vente.
              </p>
              <TransactionCalculator />
            </section>

            {/* Notre processus */}
            <section className="mb-16" id="processus">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Notre processus de transaction</h2>
              <div className="space-y-8">
                {[
                  {
                    step: "1",
                    title: "Évaluation et stratégie",
                    description:
                      "Nous commençons par une estimation précise de votre bien ou une analyse détaillée de vos besoins d'achat, puis nous élaborons une stratégie personnalisée.",
                  },
                  {
                    step: "2",
                    title: "Marketing et recherche",
                    description:
                      "Pour les vendeurs, nous mettons en place une stratégie marketing ciblée. Pour les acheteurs, nous effectuons une recherche personnalisée selon vos critères.",
                  },
                  {
                    step: "3",
                    title: "Visites et sélection",
                    description:
                      "Organisation de visites qualifiées pour les vendeurs, présélection rigoureuse des biens pour les acheteurs.",
                  },
                  {
                    step: "4",
                    title: "Négociation et offre",
                    description:
                      "Nous négocions les meilleures conditions pour vous, que vous soyez vendeur ou acheteur, et sécurisons l'offre d'achat.",
                  },
                  {
                    step: "5",
                    title: "Accompagnement jusqu'à la signature",
                    description:
                      "Nous vous accompagnons dans toutes les démarches administratives et juridiques jusqu'à la signature définitive chez le notaire.",
                  },
                ].map((step, index) => (
                  <div key={index} className="flex gap-6">
                    <div className="flex-shrink-0 w-12 h-12 rounded-full bg-[#002395] text-white flex items-center justify-center text-xl font-bold">
                      {step.step}
                    </div>
                    <div>
                      <h3 className="text-xl font-semibold mb-2">{step.title}</h3>
                      <p className="text-gray-700">{step.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* Témoignages */}
            <section className="mb-16" id="temoignages">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Ce que nos clients disent de nous</h2>
              <div className="grid md:grid-cols-2 gap-6">
                {[
                  {
                    name: "Sophie M.",
                    role: "Vendeuse à Paris 16e",
                    testimonial:
                      "J'ai vendu mon appartement avec DOGES en un temps record et à un prix supérieur à mes attentes. Leur stratégie marketing et leur réseau d'acheteurs ont fait toute la différence.",
                  },
                  {
                    name: "Thomas L.",
                    role: "Acheteur à Paris 9e",
                    testimonial:
                      "Après plusieurs mois de recherche infructueuse, DOGES m'a trouvé l'appartement idéal en seulement 3 semaines. Leur connaissance du marché parisien est impressionnante.",
                  },
                  {
                    name: "Marie et Pierre D.",
                    role: "Vendeurs à Neuilly-sur-Seine",
                    testimonial:
                      "Un accompagnement de A à Z pour la vente de notre maison familiale. Professionnalisme, réactivité et conseils avisés ont rendu cette transaction sereine malgré l'aspect émotionnel.",
                  },
                  {
                    name: "Jean-Philippe B.",
                    role: "Investisseur",
                    testimonial:
                      "Je travaille avec DOGES depuis plusieurs années pour mes investissements locatifs. Leur expertise me permet d'identifier les biens à fort potentiel et de négocier aux meilleures conditions.",
                  },
                ].map((testimonial, index) => (
                  <div key={index} className="bg-white p-6 rounded-xl shadow-sm border border-gray-100">
                    <div className="flex items-center mb-4">
                      <div className="w-12 h-12 rounded-full bg-gray-200 flex items-center justify-center text-gray-500 font-bold">
                        {testimonial.name.charAt(0)}
                      </div>
                      <div className="ml-4">
                        <h4 className="font-semibold">{testimonial.name}</h4>
                        <p className="text-sm text-gray-600">{testimonial.role}</p>
                      </div>
                    </div>
                    <p className="italic text-gray-700">"{testimonial.testimonial}"</p>
                  </div>
                ))}
              </div>
            </section>

            {/* FAQ */}
            <section className="mb-16" id="faq">
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">Questions fréquentes</h2>
              <div className="space-y-6">
                {[
                  {
                    question: "Quels sont les frais liés à une transaction immobilière ?",
                    answer:
                      "Les frais de transaction immobilière comprennent principalement les frais de notaire (7-8% pour l'ancien, 2-3% pour le neuf), les honoraires d'agence (généralement 4-6% du prix de vente), et éventuellement des frais de dossier bancaire si vous contractez un prêt. Notre calculateur vous permet d'estimer précisément ces frais selon votre situation.",
                  },
                  {
                    question: "Combien de temps faut-il pour vendre un bien à Paris ?",
                    answer:
                      "Le délai moyen de vente à Paris varie entre 45 et 90 jours, selon le type de bien, sa localisation et son prix. Chez DOGES, grâce à notre réseau d'acquéreurs qualifiés et notre stratégie marketing ciblée, nous parvenons souvent à réduire ce délai. Chaque bien étant unique, nous vous proposons une estimation de délai personnalisée lors de notre première rencontre.",
                  },
                  {
                    question: "Comment se déroule une visite avec DOGES ?",
                    answer:
                      "Nos visites sont toujours préparées en amont. Pour les vendeurs, nous valorisons votre bien et qualifions les acquéreurs avant la visite. Pour les acheteurs, nous sélectionnons rigoureusement les biens correspondant à vos critères et vous accompagnons personnellement lors des visites pour vous conseiller et répondre à toutes vos questions. Nous pouvons également organiser des visites virtuelles pour un premier aperçu.",
                  },
                  {
                    question: "Quels documents faut-il préparer pour vendre son bien ?",
                    answer:
                      "Pour vendre votre bien, vous devez réunir : le titre de propriété, les diagnostics techniques obligatoires (DPE, amiante, plomb, etc.), le dernier avis de taxe foncière, le règlement de copropriété et les 3 derniers PV d'assemblée générale pour un appartement, ainsi que les factures des travaux importants réalisés. Notre équipe vous accompagne dans la préparation de ce dossier et peut organiser la réalisation des diagnostics obligatoires.",
                  },
                  {
                    question: "Proposez-vous des services pour les acquéreurs étrangers ?",
                    answer:
                      "Oui, nous disposons d'une expertise spécifique pour accompagner les acquéreurs internationaux. Notre équipe multilingue peut vous assister dans toutes les étapes de votre acquisition à Paris : recherche personnalisée, visites virtuelles, conseil fiscal et juridique adapté aux non-résidents, et coordination avec les notaires. Nous pouvons également vous mettre en relation avec des partenaires bancaires spécialisés dans les financements pour les non-résidents.",
                  },
                ].map((item, index) => (
                  <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
                    <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
                    <p className="text-gray-700">{item.answer}</p>
                  </div>
                ))}
              </div>
              <div className="mt-6 text-center">
                <Link href="/faq" className="text-[#002395] font-medium hover:underline inline-flex items-center">
                  Voir toutes les questions fréquentes
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Link>
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="lg:w-1/3">
            {/* Table des matières */}
            <div className="mb-6">
              <TableOfContentsWidget items={tocItems} />
            </div>

            {/* Calculateur rapide */}
            <div className="mb-6">
              <SidebarCalculator />
            </div>

            {/* Widget de contact rapide */}
            <ContactWidget />

            {/* Services associés */}
            <RelatedServicesWidget services={relatedServices} />
          </div>
        </div>

        {/* CTA Section */}
        <div className="mt-12">
          <div className="bg-[#002395] text-white rounded-lg p-6 shadow-lg">
            <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-4">
              <div className="space-y-2">
                <h3 className="text-xl font-bold">Prêt à vendre ou acheter un bien immobilier ?</h3>
                <p className="text-white/90 max-w-2xl">
                  Nos experts vous accompagnent à chaque étape de votre projet immobilier.
                </p>
              </div>
              <a
                href="/contact"
                className="bg-white text-[#002395] px-6 py-3 rounded-md font-medium hover:bg-gray-100 transition-colors whitespace-nowrap"
              >
                Prendre rendez-vous
              </a>
            </div>
          </div>
        </div>
      </div>
    </>
  )
}
