import type { Metadata } from "next"
import Image from "next/image"
import { Check, MapPin, Search, Star, Building2, Shield, Clock } from "lucide-react"
import { Button } from "@/components/ui/button"
import ServicePageLayout from "@/components/service-page-layout"

// Métadonnées SEO optimisées
export const metadata: Metadata = {
  title: "Trouver un Gestionnaire Immobilier à Paris et en Île-de-France | DOGES",
  description:
    "Comment trouver un bon gestionnaire immobilier à Paris et en Île-de-France ? Critères de choix, conseils d'experts et recherche personnalisée pour sélectionner le meilleur administrateur de biens.",
  keywords:
    "trouver gestionnaire immobilier, choisir administrateur de biens, meilleur gestionnaire locatif Paris, comparatif gestionnaires immobiliers, critères choix gestionnaire immobilier",
}

export default function TrouverGestionnaireImmobilierPage() {
  // Définir les éléments pour la table des matières
  const tableOfContentsItems = [
    { id: "pourquoi-gestionnaire", title: "Pourquoi faire appel à un gestionnaire immobilier ?" },
    { id: "criteres-choix", title: "Les critères pour bien choisir son gestionnaire" },
    { id: "recherche-personnalisee", title: "Recherche personnalisée de gestionnaire immobilier" },
    { id: "questions-poser", title: "Les questions à poser à votre futur gestionnaire" },
    { id: "avantages-doges", title: "Les avantages de choisir DOGES comme gestionnaire" },
    { id: "faq", title: "Questions fréquentes" },
  ]

  // Définir les services associés
  const relatedServices = [
    {
      title: "Gestion locative Paris",
      description: "Notre service de gestion locative spécialisé pour Paris",
      link: "/services/gestion-locative-paris",
    },
    {
      title: "Gestion locative Île-de-France",
      description: "Gestion de biens immobiliers en Île-de-France",
      link: "/services/gestion-locative-ile-de-france",
    },
    {
      title: "Garantie loyers impayés",
      description: "Protection contre les risques d'impayés",
      link: "/services/garantie-loyers-impayes",
    },
  ]

  // Propriétés pour le CTA
  const ctaProps = {
    title: "Besoin d'aide pour trouver votre gestionnaire immobilier ?",
    description: "Contactez-nous pour un accompagnement personnalisé dans votre recherche de gestionnaire immobilier.",
    buttonText: "Demander un devis personnalisé",
    buttonLink: "/contact",
  }

  return (
    <ServicePageLayout
      title="Trouver un Gestionnaire Immobilier"
      description="Comment choisir le bon administrateur de biens pour votre patrimoine immobilier"
      breadcrumbItems={[
        { name: "Accueil", url: "/" },
        { name: "Trouver un Gestionnaire Immobilier", url: "/trouver-gestionnaire-immobilier" },
      ]}
      tableOfContentsItems={tableOfContentsItems}
      headerImage="https://images.unsplash.com/photo-1560518883-ce09059eeffa?q=80&w=2073&auto=format&fit=crop"
      relatedServices={relatedServices}
      ctaProps={ctaProps}
    >
      {/* Pourquoi faire appel à un gestionnaire immobilier */}
      <section id="pourquoi-gestionnaire" className="mb-12">
        <h2 className="text-3xl font-bold mb-6 text-[#002395]">
          Pourquoi faire appel à un gestionnaire immobilier professionnel ?
        </h2>
        <p className="text-lg text-gray-700 mb-6">
          La gestion d'un bien immobilier demande du temps, des connaissances spécifiques et une veille constante sur
          les évolutions législatives. Faire appel à un <strong>gestionnaire immobilier professionnel</strong> vous
          permet de déléguer ces responsabilités à un expert du secteur.
        </p>
        <p className="text-lg text-gray-700 mb-6">
          Un bon administrateur de biens vous fait gagner du temps et de l'argent en optimisant la gestion de votre
          patrimoine immobilier. Il vous protège également contre les risques locatifs et vous assure une tranquillité
          d'esprit.
        </p>
        <div className="space-y-4">
          {[
            "Gain de temps considérable pour le propriétaire",
            "Expertise juridique et technique à votre service",
            "Optimisation de la rentabilité de votre bien",
            "Protection contre les risques locatifs (impayés, dégradations)",
            "Gestion administrative complète et simplifiée",
            "Valorisation de votre patrimoine immobilier",
          ].map((avantage, index) => (
            <div key={index} className="flex items-start">
              <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                <Check className="h-4 w-4 text-white" />
              </div>
              <p>{avantage}</p>
            </div>
          ))}
        </div>
        <div className="mt-8 relative h-[400px] rounded-2xl overflow-hidden">
          <Image
            src="https://images.unsplash.com/photo-1560518883-ce09059eeffa?q=80&w=2073&auto=format&fit=crop"
            alt="Gestionnaire immobilier professionnel"
            fill
            className="object-cover"
          />
        </div>
      </section>

      {/* Les critères pour bien choisir son gestionnaire */}
      <section id="criteres-choix" className="mb-12">
        <h2 className="text-3xl font-bold mb-8 text-[#002395]">
          Les critères pour bien choisir son gestionnaire immobilier
        </h2>

        <div className="grid md:grid-cols-2 gap-6">
          {[
            {
              title: "Expertise et expérience",
              description:
                "Vérifiez l'ancienneté de l'agence, son expertise sur votre secteur géographique et les qualifications de ses collaborateurs. Un gestionnaire expérimenté connaît parfaitement le marché local et ses spécificités.",
              icon: <Building2 className="h-10 w-10 text-white" />,
            },
            {
              title: "Garanties professionnelles",
              description:
                "Assurez-vous que le gestionnaire possède une carte professionnelle valide, une garantie financière suffisante et une assurance responsabilité civile professionnelle. Ces éléments sont obligatoires et garantissent son sérieux.",
              icon: <Shield className="h-10 w-10 text-white" />,
            },
            {
              title: "Transparence des honoraires",
              description:
                "Les honoraires doivent être clairement détaillés et compétitifs. Méfiez-vous des tarifs anormalement bas qui peuvent cacher un service de moindre qualité ou des frais annexes importants.",
              icon: <Building2 className="h-10 w-10 text-white" />,
            },
            {
              title: "Réactivité et disponibilité",
              description:
                "Un bon gestionnaire doit être réactif et disponible pour répondre à vos questions et gérer les urgences. Testez sa réactivité dès vos premiers échanges pour vous faire une idée.",
              icon: <Clock className="h-10 w-10 text-white" />,
            },
            {
              title: "Outils digitaux",
              description:
                "Privilégiez un gestionnaire qui propose des outils digitaux modernes : espace client en ligne, reporting régulier, communication facilitée. Ces outils témoignent d'une gestion efficace et transparente.",
              icon: <Building2 className="h-10 w-10 text-white" />,
            },
            {
              title: "Avis et recommandations",
              description:
                "Consultez les avis clients et demandez des références. L'expérience d'autres propriétaires est précieuse pour évaluer la qualité du service et la satisfaction globale des clients.",
              icon: <Star className="h-10 w-10 text-white" />,
            },
          ].map((critere, index) => (
            <div key={index} className="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
              <div className="bg-gradient-to-r from-[#002395] to-[#001875] w-16 h-16 rounded-full flex items-center justify-center mb-6">
                {critere.icon}
              </div>
              <h3 className="text-xl font-bold mb-4">{critere.title}</h3>
              <p className="text-gray-700">{critere.description}</p>
            </div>
          ))}
        </div>
      </section>

      {/* Recherche personnalisée */}
      <section id="recherche-personnalisee" className="mb-12">
        <h2 className="text-3xl font-bold mb-8 text-[#002395]">Recherche personnalisée de gestionnaire immobilier</h2>

        <div className="bg-white rounded-xl shadow-lg p-8 border border-gray-100">
          <p className="text-lg text-gray-700 mb-8 text-center">
            Utilisez notre outil de recherche pour trouver le gestionnaire immobilier idéal pour votre bien. Précisez
            vos critères et nous vous proposerons les meilleures options.
          </p>

          <form className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-2">
                <label className="block font-medium">Type de bien</label>
                <select className="w-full p-3 border rounded-lg">
                  <option value="">Sélectionnez</option>
                  <option value="appartement">Appartement</option>
                  <option value="maison">Maison</option>
                  <option value="immeuble">Immeuble entier</option>
                  <option value="commerce">Local commercial</option>
                  <option value="bureau">Bureau</option>
                </select>
              </div>
              <div className="space-y-2">
                <label className="block font-medium">Localisation</label>
                <div className="relative">
                  <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                  <input
                    type="text"
                    placeholder="Code postal ou ville"
                    className="w-full p-3 pl-10 border rounded-lg"
                  />
                </div>
              </div>
            </div>

            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-2">
                <label className="block font-medium">Service recherché</label>
                <select className="w-full p-3 border rounded-lg">
                  <option value="">Sélectionnez</option>
                  <option value="gestion-locative">Gestion locative</option>
                  <option value="syndic">Syndic de copropriété</option>
                  <option value="transaction">Transaction immobilière</option>
                  <option value="conseil">Conseil patrimonial</option>
                  <option value="tous">Tous les services</option>
                </select>
              </div>
              <div className="space-y-2">
                <label className="block font-medium">Budget mensuel (honoraires)</label>
                <select className="w-full p-3 border rounded-lg">
                  <option value="">Sélectionnez</option>
                  <option value="moins-50">Moins de 50€/mois</option>
                  <option value="50-100">Entre 50€ et 100€/mois</option>
                  <option value="100-200">Entre 100€ et 200€/mois</option>
                  <option value="plus-200">Plus de 200€/mois</option>
                </select>
              </div>
            </div>

            <div className="space-y-2">
              <label className="block font-medium">Critères importants pour vous</label>
              <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                {[
                  "Expertise locale",
                  "Réactivité",
                  "Transparence",
                  "Outils digitaux",
                  "Garantie loyers impayés",
                  "Gestion technique",
                ].map((critere, index) => (
                  <div key={index} className="flex items-center">
                    <input type="checkbox" id={`critere-${index}`} className="mr-2" />
                    <label htmlFor={`critere-${index}`}>{critere}</label>
                  </div>
                ))}
              </div>
            </div>

            <div className="flex justify-center">
              <Button className="bg-gradient-to-r from-[#002395] to-[#001875] px-8 py-3">
                <Search className="mr-2 h-5 w-5" />
                Rechercher un gestionnaire
              </Button>
            </div>
          </form>
        </div>
      </section>

      {/* Les questions à poser */}
      <section id="questions-poser" className="mb-12">
        <h2 className="text-3xl font-bold mb-8 text-[#002395]">
          Les questions à poser à votre futur gestionnaire immobilier
        </h2>

        <div className="grid md:grid-cols-2 gap-6">
          {[
            {
              question: "Quels sont vos honoraires détaillés ?",
              description:
                "Demandez un détail complet des honoraires : gestion courante, mise en location, assurances, etc. Méfiez-vous des tarifs trop bas qui cachent souvent des frais annexes.",
            },
            {
              question: "Combien de biens gérez-vous actuellement ?",
              description:
                "Cette question vous permet d'évaluer l'expérience du gestionnaire mais aussi sa charge de travail. Un portefeuille trop important peut nuire à la qualité du service.",
            },
            {
              question: "Comment sélectionnez-vous les locataires ?",
              description:
                "Le processus de sélection des locataires est crucial pour éviter les impayés. Assurez-vous que le gestionnaire effectue des vérifications rigoureuses.",
            },
            {
              question: "Quel est votre délai moyen de relocation ?",
              description:
                "Cette information vous donne une idée de l'efficacité du gestionnaire et de sa capacité à limiter la vacance locative, qui peut être coûteuse.",
            },
            {
              question: "Comment gérez-vous les travaux et interventions ?",
              description:
                "Demandez des précisions sur le processus de gestion des travaux : devis comparatifs, seuil d'autorisation, suivi des interventions, etc.",
            },
            {
              question: "Quels outils de reporting proposez-vous ?",
              description:
                "Un bon gestionnaire doit vous fournir des rapports réguliers et détaillés sur la gestion de votre bien : loyers, charges, travaux, etc.",
            },
            {
              question: "Comment gérez-vous les impayés ?",
              description:
                "La gestion des impayés est un point crucial. Renseignez-vous sur les procédures mises en place et les délais d'intervention en cas de retard de paiement.",
            },
            {
              question: "Proposez-vous une garantie loyers impayés ?",
              description:
                "Cette assurance est importante pour sécuriser vos revenus locatifs. Demandez les conditions et le coût de cette garantie.",
            },
          ].map((item, index) => (
            <div key={index} className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
              <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
              <p className="text-gray-700">{item.description}</p>
            </div>
          ))}
        </div>
      </section>

      {/* Les avantages de choisir DOGES */}
      <section id="avantages-doges" className="mb-12">
        <h2 className="text-3xl font-bold mb-6 text-[#002395]">
          Les avantages de choisir DOGES comme gestionnaire immobilier
        </h2>
        <p className="text-lg text-gray-700 mb-6">
          Avec plus de 25 ans d'expérience dans l'administration de biens à Paris et en Île-de-France, DOGES s'impose
          comme un partenaire de confiance pour la gestion de votre patrimoine immobilier.
        </p>
        <div className="space-y-6">
          {[
            {
              title: "Expertise locale inégalée",
              description:
                "Notre connaissance approfondie du marché immobilier parisien nous permet d'optimiser la gestion de votre bien selon sa localisation spécifique.",
            },
            {
              title: "Équipe dédiée et réactive",
              description:
                "Chaque client bénéficie d'un gestionnaire dédié, disponible et réactif, pour un suivi personnalisé de son patrimoine.",
            },
            {
              title: "Transparence totale",
              description:
                "Nous vous proposons un reporting détaillé et régulier, ainsi qu'un accès à votre espace client en ligne pour suivre la gestion de votre bien en temps réel.",
            },
            {
              title: "Solutions sur mesure",
              description:
                "Nos forfaits sont adaptables à vos besoins spécifiques, que vous possédiez un studio ou un portefeuille immobilier conséquent.",
            },
            {
              title: "Outils digitaux innovants",
              description:
                "Notre plateforme digitale vous permet de suivre la gestion de votre bien en temps réel et de communiquer facilement avec votre gestionnaire.",
            },
          ].map((avantage, index) => (
            <div key={index} className="flex items-start">
              <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                <Check className="h-4 w-4 text-white" />
              </div>
              <div>
                <h3 className="font-semibold">{avantage.title}</h3>
                <p className="text-gray-700">{avantage.description}</p>
              </div>
            </div>
          ))}
        </div>
        <div className="mt-8 relative h-[400px] rounded-2xl overflow-hidden">
          <Image
            src="https://images.unsplash.com/photo-1560518883-b8e76de6e641?q=80&w=2073&auto=format&fit=crop"
            alt="DOGES gestionnaire immobilier"
            fill
            className="object-cover"
          />
        </div>
      </section>

      {/* Comparatif des services */}
      <section className="mb-12">
        <h2 className="text-3xl font-bold mb-8 text-[#002395]">Comparatif des services de gestion immobilière</h2>

        <div className="overflow-x-auto">
          <table className="w-full bg-white rounded-xl shadow-sm">
            <thead>
              <tr className="bg-[#002395] text-white">
                <th className="py-4 px-6 text-left">Services</th>
                <th className="py-4 px-6 text-center">Gestion par le propriétaire</th>
                <th className="py-4 px-6 text-center">Gestionnaire standard</th>
                <th className="py-4 px-6 text-center">DOGES</th>
              </tr>
            </thead>
            <tbody>
              {[
                {
                  service: "Recherche et sélection de locataires",
                  proprio: "Limitée",
                  standard: "Basique",
                  doges: "Premium",
                },
                {
                  service: "Gestion des loyers et charges",
                  proprio: "Manuelle",
                  standard: "Automatisée",
                  doges: "Automatisée + suivi personnalisé",
                },
                {
                  service: "Gestion technique",
                  proprio: "Limitée",
                  standard: "Standard",
                  doges: "Réseau d'artisans qualifiés",
                },
                {
                  service: "Garantie loyers impayés",
                  proprio: "Non",
                  standard: "Option payante",
                  doges: "Incluse dans certains forfaits",
                },
                {
                  service: "Reporting et transparence",
                  proprio: "Manuelle",
                  standard: "Périodique",
                  doges: "Temps réel + espace client",
                },
                {
                  service: "Conseil juridique et fiscal",
                  proprio: "Non",
                  standard: "Basique",
                  doges: "Expert dédié",
                },
                {
                  service: "Réactivité",
                  proprio: "Variable",
                  standard: "Moyenne",
                  doges: "Garantie sous 24h",
                },
              ].map((row, index) => (
                <tr key={index} className={index % 2 === 0 ? "bg-gray-50" : "bg-white"}>
                  <td className="py-4 px-6 font-medium">{row.service}</td>
                  <td className="py-4 px-6 text-center">{row.proprio}</td>
                  <td className="py-4 px-6 text-center">{row.standard}</td>
                  <td className="py-4 px-6 text-center font-semibold text-[#002395]">{row.doges}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </section>

      {/* FAQ */}
      <section id="faq" className="mb-12">
        <h2 className="text-3xl font-bold mb-8 text-[#002395]">
          Questions fréquentes sur la recherche d'un gestionnaire immobilier
        </h2>

        <div className="space-y-6">
          {[
            {
              question: "Combien coûte un gestionnaire immobilier à Paris ?",
              answer:
                "Les honoraires d'un gestionnaire immobilier à Paris varient généralement entre 6% et 10% HT des loyers encaissés pour la gestion locative standard. Ce pourcentage peut varier selon le type de bien, sa localisation et les services inclus. Chez DOGES, nos tarifs débutent à 6% HT avec différentes formules adaptées à vos besoins spécifiques. Des frais supplémentaires peuvent s'appliquer pour la mise en location (généralement un mois de loyer) ou pour des services spécifiques comme la garantie loyers impayés (environ 2,5% à 3,5% des loyers).",
            },
            {
              question: "Quelle est la différence entre un gestionnaire immobilier et un syndic de copropriété ?",
              answer:
                "Un gestionnaire immobilier (ou administrateur de biens) s'occupe de la gestion locative d'un bien pour le compte d'un propriétaire individuel. Ses missions incluent la recherche de locataires, l'encaissement des loyers, la gestion des travaux, etc. Un syndic de copropriété, quant à lui, gère les parties communes d'un immeuble pour le compte de l'ensemble des copropriétaires. Il s'occupe de l'entretien des parties communes, de la comptabilité de la copropriété, de l'organisation des assemblées générales, etc. DOGES propose ces deux services complémentaires, ce qui peut être un avantage si vous possédez un bien dans une copropriété que nous gérons déjà.",
            },
            {
              question: "Comment vérifier la fiabilité d'un gestionnaire immobilier ?",
              answer:
                "Pour vérifier la fiabilité d'un gestionnaire immobilier, plusieurs éléments sont à contrôler : sa carte professionnelle (obligatoire et délivrée par la CCI), sa garantie financière (qui protège les fonds qu'il gère), son assurance responsabilité civile professionnelle, son ancienneté sur le marché, les avis clients et les recommandations. N'hésitez pas à demander des références de clients actuels et à vérifier les avis en ligne. Chez DOGES, nous sommes transparents sur nos garanties et pouvons vous mettre en relation avec des clients satisfaits de nos services depuis de nombreuses années.",
            },
            {
              question: "Puis-je changer de gestionnaire immobilier facilement ?",
              answer:
                "Oui, vous pouvez changer de gestionnaire immobilier en respectant les conditions de résiliation prévues dans votre mandat de gestion (généralement un préavis de 3 mois). Le processus est relativement simple : vous devez envoyer une lettre recommandée avec accusé de réception à votre gestionnaire actuel pour résilier le mandat, puis signer un nouveau mandat avec DOGES. Nous nous occupons ensuite de toutes les démarches pour récupérer votre dossier auprès de l'ancien gestionnaire (documents, dépôt de garantie, etc.) et assurer une transition en douceur. Ce changement n'a aucun impact sur votre locataire actuel, dont le bail reste valide.",
            },
          ].map((item, index) => (
            <div key={index} className="bg-white rounded-lg shadow-sm p-6 border border-gray-100">
              <h3 className="text-lg font-semibold mb-2">{item.question}</h3>
              <p className="text-gray-700">{item.answer}</p>
            </div>
          ))}
        </div>
      </section>
    </ServicePageLayout>
  )
}
