"use client"

import type React from "react"

import { useState } from "react"
import { Calendar } from "@/components/ui/calendar"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { fr } from "date-fns/locale"
import { ArrowRight, CalendarIcon } from "lucide-react"

export default function AppointmentSystem() {
  const [date, setDate] = useState<Date | undefined>(undefined)
  const [timeSlot, setTimeSlot] = useState<string | null>(null)
  const [step, setStep] = useState(1)
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    service: "",
    message: "",
  })

  // Générer des créneaux horaires disponibles (simulés)
  const generateTimeSlots = () => {
    // Dans une vraie application, ces créneaux seraient récupérés depuis une API
    const slots = []
    for (let hour = 9; hour <= 17; hour++) {
      if (hour !== 12 && hour !== 13) {
        // Pause déjeuner
        slots.push(`${hour}:00`)
        if (hour !== 17) slots.push(`${hour}:30`)
      }
    }
    return slots
  }

  const availableTimeSlots = generateTimeSlots()

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target
    setFormData((prev) => ({ ...prev, [name]: value }))
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    // Simuler l'envoi des données
    console.log("Rendez-vous confirmé:", { date, timeSlot, ...formData })
    alert("Votre rendez-vous a été confirmé. Vous recevrez un email de confirmation.")
    // Réinitialiser le formulaire
    setDate(undefined)
    setTimeSlot(null)
    setStep(1)
    setFormData({
      name: "",
      email: "",
      phone: "",
      service: "",
      message: "",
    })
  }

  const nextStep = () => {
    if (step === 1 && !date) {
      alert("Veuillez sélectionner une date")
      return
    }
    if (step === 2 && !timeSlot) {
      alert("Veuillez sélectionner un créneau horaire")
      return
    }
    setStep(step + 1)
  }

  const prevStep = () => {
    setStep(step - 1)
  }

  return (
    <Dialog>
      <DialogTrigger asChild>
        <Button className="bg-gradient-to-r from-[#002395] to-[#001875]">
          <CalendarIcon className="mr-2 h-4 w-4" />
          Prendre rendez-vous
        </Button>
      </DialogTrigger>
      <DialogContent className="sm:max-w-[600px]">
        <DialogHeader>
          <DialogTitle>Prendre rendez-vous avec un conseiller</DialogTitle>
        </DialogHeader>

        {step === 1 && (
          <div className="space-y-4">
            <p className="text-sm text-gray-500">Étape 1/3: Sélectionnez une date</p>
            <div className="flex justify-center">
              <Calendar
                mode="single"
                selected={date}
                onSelect={setDate}
                locale={fr}
                className="rounded-md border"
                disabled={(date) => {
                  // Désactiver les week-ends et les dates passées
                  const day = date.getDay()
                  const today = new Date()
                  today.setHours(0, 0, 0, 0)
                  return day === 0 || day === 6 || date < today
                }}
              />
            </div>
            <div className="flex justify-end">
              <Button onClick={nextStep} className="bg-gradient-to-r from-[#002395] to-[#001875]">
                Suivant
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
            </div>
          </div>
        )}

        {step === 2 && (
          <div className="space-y-4">
            <p className="text-sm text-gray-500">Étape 2/3: Choisissez un créneau horaire</p>
            <div className="grid grid-cols-3 gap-2">
              {availableTimeSlots.map((slot) => (
                <button
                  key={slot}
                  className={`p-2 border rounded-md text-center transition-colors ${
                    timeSlot === slot ? "bg-[#002395] text-white" : "hover:bg-gray-100"
                  }`}
                  onClick={() => setTimeSlot(slot)}
                >
                  {slot}
                </button>
              ))}
            </div>
            <div className="flex justify-between">
              <Button variant="outline" onClick={prevStep}>
                Retour
              </Button>
              <Button onClick={nextStep} className="bg-gradient-to-r from-[#002395] to-[#001875]">
                Suivant
                <ArrowRight className="ml-2 h-4 w-4" />
              </Button>
            </div>
          </div>
        )}

        {step === 3 && (
          <form onSubmit={handleSubmit} className="space-y-4">
            <p className="text-sm text-gray-500">Étape 3/3: Vos informations</p>

            <div className="space-y-2">
              <Label htmlFor="name">Nom complet</Label>
              <Input id="name" name="name" value={formData.name} onChange={handleInputChange} required />
            </div>

            <div className="space-y-2">
              <Label htmlFor="email">Email</Label>
              <Input
                id="email"
                name="email"
                type="email"
                value={formData.email}
                onChange={handleInputChange}
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="phone">Téléphone</Label>
              <Input id="phone" name="phone" type="tel" value={formData.phone} onChange={handleInputChange} required />
            </div>

            <div className="space-y-2">
              <Label htmlFor="service">Service concerné</Label>
              <select
                id="service"
                name="service"
                value={formData.service}
                onChange={handleInputChange}
                className="w-full h-9 rounded-md border border-input bg-transparent px-3 py-1 text-sm shadow-sm transition-colors"
                required
              >
                <option value="">Sélectionnez un service</option>
                <option value="gestion-locative">Gestion locative</option>
                <option value="transaction">Transaction immobilière</option>
                <option value="gli">Garantie loyers impayés</option>
                <option value="pno">Assurance PNO</option>
                <option value="syndic">Syndic de copropriété</option>
              </select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="message">Message (optionnel)</Label>
              <textarea
                id="message"
                name="message"
                value={formData.message}
                onChange={handleInputChange}
                className="w-full min-h-[100px] rounded-md border border-input bg-transparent px-3 py-2 text-sm shadow-sm transition-colors"
                placeholder="Précisez votre demande..."
              ></textarea>
            </div>

            <div className="pt-2 text-sm">
              <p>
                <strong>Date sélectionnée:</strong> {date?.toLocaleDateString("fr-FR")}
              </p>
              <p>
                <strong>Horaire:</strong> {timeSlot}
              </p>
            </div>

            <div className="flex justify-between">
              <Button variant="outline" onClick={prevStep}>
                Retour
              </Button>
              <Button type="submit" className="bg-gradient-to-r from-[#002395] to-[#001875]">
                Confirmer le rendez-vous
              </Button>
            </div>
          </form>
        )}
      </DialogContent>
    </Dialog>
  )
}
