"use client"

import type React from "react"
import { TableOfContents } from "@/components/table-of-contents"
import { ShareButtons } from "@/components/share-buttons"
import { RelatedArticlesCards } from "@/components/related-articles-cards"
import { ConsultationCTA } from "@/components/consultation-cta"
import { BlogAuthor } from "@/components/blog-author"
import BlogSidebarWidgets from "@/components/blog-sidebar-widgets"
import BlogHeader from "@/components/blog-header"

interface Author {
  name: string
  role?: string
  title?: string
  description?: string
  bio?: string
  image?: string
  linkedin?: string
}

interface RelatedArticle {
  slug: string
  title: string
  description: string
  image?: string
  date?: string
  category?: string
}

interface BlogArticleLayoutProps {
  children: React.ReactNode
  // Allow both direct props and metadata object
  title?: string
  date?: string
  readingTime?: string
  category?: string
  image?: string
  author?: Author
  relatedArticles?: RelatedArticle[]
  metadata?: {
    title: string
    description: string
    author: Author
    date: string
    readingTime: string
    category: string
    image?: string
    relatedArticles?: RelatedArticle[]
  }
  tableOfContents?:
    | Array<{
        id: string
        text: string
        level: number
      }>
    | {
        items: Array<{
          title: string
          url: string
          items?: Array<{
            title: string
            url: string
          }>
        }>
      }
  previousArticle?: {
    slug: string
    title: string
  }
  nextArticle?: {
    slug: string
    title: string
  }
}

export function BlogArticleLayout({
  children,
  title,
  date,
  readingTime,
  category,
  image,
  author,
  relatedArticles,
  metadata,
  tableOfContents,
  previousArticle,
  nextArticle,
}: BlogArticleLayoutProps) {
  const currentUrl = typeof window !== "undefined" ? window.location.href : ""

  // Use metadata if provided, otherwise use direct props
  const articleTitle = metadata?.title || title || ""
  const articleDate = metadata?.date || date || ""
  const articleReadingTime = metadata?.readingTime || readingTime || ""
  const articleCategory = metadata?.category || category || ""
  const articleImage = metadata?.image || image || ""
  const articleAuthor = metadata?.author || author || { name: "" }
  const articleRelatedArticles = metadata?.relatedArticles || relatedArticles || []

  // Convert tableOfContents au format attendu
  const formattedTableOfContents = Array.isArray(tableOfContents)
    ? tableOfContents.map((item) => ({
        id: item.id,
        text: item.text,
        level: item.level,
        url: `#${item.id}`,
      }))
    : tableOfContents && "items" in tableOfContents
      ? tableOfContents.items.map((item) => ({
          id: item.url.replace("#", ""),
          text: item.title,
          level: item.items ? 2 : 3,
          url: item.url,
        }))
      : tableOfContents

  return (
    <>
      {/* BlogHeader component */}
      <BlogHeader
        title={articleTitle}
        date={articleDate}
        category={articleCategory}
        readingTime={articleReadingTime}
        image={articleImage}
      />

      <div className="container mx-auto px-4 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-12 gap-8">
          {/* Article content */}
          <div className="lg:col-span-8">
            <article className="prose prose-slate max-w-none">{children}</article>

            {/* Utility buttons - moved to bottom of article */}
            <ShareButtons title={articleTitle} url={currentUrl} className="mt-8" />

            <div className="mt-12">
              <ConsultationCTA
                title="Besoin d'aide pour votre investissement immobilier ?"
                description="Nos experts sont disponibles pour vous accompagner dans tous vos projets immobiliers en Île-de-France."
                buttonText="Prendre rendez-vous"
                buttonLink="/contact"
              />
            </div>

            {articleRelatedArticles && articleRelatedArticles.length > 0 && (
              <div className="mt-12">
                <h2 className="text-2xl font-bold mb-6">Articles similaires</h2>
                <RelatedArticlesCards articles={articleRelatedArticles} />
              </div>
            )}
          </div>

          {/* Sidebar */}
          <div className="lg:col-span-4">
            <div className="sticky top-24 space-y-8">
              {/* Table of contents - moved to first position */}
              {formattedTableOfContents &&
                ((Array.isArray(formattedTableOfContents) && formattedTableOfContents.length > 0) ||
                  (formattedTableOfContents?.items && formattedTableOfContents.items.length > 0)) && (
                  <div className="p-4 bg-gray-50 rounded-lg">
                    <h3 className="font-bold text-lg mb-4">Sommaire</h3>
                    <TableOfContents
                      items={
                        Array.isArray(formattedTableOfContents)
                          ? formattedTableOfContents
                          : formattedTableOfContents && "items" in formattedTableOfContents
                            ? formattedTableOfContents.items
                            : []
                      }
                    />
                  </div>
                )}

              {/* Author information */}
              {articleAuthor && (
                <div className="p-4 bg-gray-50 rounded-lg">
                  <h3 className="font-bold text-lg mb-4">À propos de l'auteur</h3>
                  <BlogAuthor
                    name={articleAuthor.name}
                    title={articleAuthor.title || articleAuthor.role || ""}
                    description={articleAuthor.description || articleAuthor.bio || ""}
                    image={articleAuthor.image}
                    linkedin={articleAuthor.linkedin}
                  />
                </div>
              )}

              {/* Additional sidebar widgets */}
              <BlogSidebarWidgets />
            </div>
          </div>
        </div>
      </div>
    </>
  )
}

export default BlogArticleLayout
