import type React from "react"
import Link from "next/link"
import BlogSidebarWidgets from "./blog-sidebar-widgets"

interface BlogLayoutProps {
  children: React.ReactNode
  categories: string[]
  currentCategory?: string
}

// Fonction améliorée pour normaliser les slugs (retirer les accents et remplacer les espaces par des tirets)
function normalizeSlug(text: string): string {
  return text
    .normalize("NFD")
    .replace(/[\u0300-\u036f]/g, "") // Supprime les accents
    .toLowerCase()
    .replace(/\s+/g, "-") // Remplace les espaces par des tirets
    .replace(/[^a-z0-9-]/g, "") // Supprime les caractères non alphanumériques (sauf tirets)
}

export default function BlogLayout({ children, categories, currentCategory }: BlogLayoutProps) {
  return (
    <div className="container mx-auto px-4 py-12">
      <div className="flex flex-col lg:flex-row gap-8">
        {/* Sidebar */}
        <div className="lg:w-1/4">
          <div className="sticky top-24">
            <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100 mb-8">
              <h3 className="text-xl font-bold mb-4">Catégories</h3>
              <ul className="space-y-2">
                <li>
                  <Link
                    href="/blog"
                    className={`block px-3 py-2 rounded-md ${
                      !currentCategory ? "bg-blue-600 text-white" : "text-gray-700 hover:bg-gray-100 transition-colors"
                    }`}
                  >
                    Tous les articles
                  </Link>
                </li>
                {categories.map((category) => (
                  <li key={category}>
                    <Link
                      href={`/blog/categories/${normalizeSlug(category)}`}
                      className={`block px-3 py-2 rounded-md ${
                        currentCategory === category
                          ? "bg-blue-600 text-white"
                          : "text-gray-700 hover:bg-gray-100 transition-colors"
                      }`}
                    >
                      {category}
                    </Link>
                  </li>
                ))}
              </ul>
            </div>

            <BlogSidebarWidgets />
          </div>
        </div>

        {/* Main content */}
        <div className="lg:w-3/4">{children}</div>
      </div>
    </div>
  )
}
