"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Package, FileText, Zap, AlertTriangle } from "lucide-react"

interface BundleInfo {
  name: string
  size: number
  gzipped: number
  type: "critical" | "important" | "optional"
  description: string
}

export default function BundleAnalyzer() {
  const [bundles, setBundles] = useState<BundleInfo[]>([])
  const [totalSize, setTotalSize] = useState(0)

  useEffect(() => {
    // Simuler l'analyse du bundle
    const mockBundles: BundleInfo[] = [
      {
        name: "Next.js Runtime",
        size: 245,
        gzipped: 89,
        type: "critical",
        description: "Runtime Next.js et React",
      },
      {
        name: "Tailwind CSS",
        size: 156,
        gzipped: 23,
        type: "critical",
        description: "Styles CSS optimisés",
      },
      {
        name: "Framer Motion",
        size: 187,
        gzipped: 52,
        type: "important",
        description: "Animations et transitions",
      },
      {
        name: "Lucide Icons",
        size: 89,
        gzipped: 21,
        type: "important",
        description: "Icônes SVG optimisées",
      },
      {
        name: "Components",
        size: 134,
        gzipped: 31,
        type: "critical",
        description: "Composants React personnalisés",
      },
      {
        name: "Utils & Hooks",
        size: 45,
        gzipped: 12,
        type: "important",
        description: "Utilitaires et hooks personnalisés",
      },
    ]

    setBundles(mockBundles)
    setTotalSize(mockBundles.reduce((acc, bundle) => acc + bundle.gzipped, 0))
  }, [])

  const getTypeColor = (type: string) => {
    switch (type) {
      case "critical":
        return "bg-red-100 text-red-800"
      case "important":
        return "bg-yellow-100 text-yellow-800"
      case "optional":
        return "bg-green-100 text-green-800"
      default:
        return "bg-gray-100 text-gray-800"
    }
  }

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "critical":
        return <AlertTriangle className="h-4 w-4" />
      case "important":
        return <Zap className="h-4 w-4" />
      case "optional":
        return <Package className="h-4 w-4" />
      default:
        return <FileText className="h-4 w-4" />
    }
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Package className="h-5 w-5" />
          <span>Analyse du Bundle JavaScript</span>
          <span className="text-sm font-normal text-gray-500">(Total: {totalSize} KB gzippé)</span>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {bundles.map((bundle, index) => (
            <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-3">
                <span className={`p-2 rounded ${getTypeColor(bundle.type)}`}>{getTypeIcon(bundle.type)}</span>
                <div>
                  <div className="font-medium">{bundle.name}</div>
                  <div className="text-sm text-gray-600">{bundle.description}</div>
                </div>
              </div>
              <div className="text-right">
                <div className="font-semibold">{bundle.gzipped} KB</div>
                <div className="text-sm text-gray-500">{bundle.size} KB brut</div>
              </div>
            </div>
          ))}
        </div>

        <div className="mt-6 p-4 bg-blue-50 rounded-lg">
          <h4 className="font-semibold text-blue-800 mb-2">Optimisations appliquées</h4>
          <ul className="text-sm text-blue-700 space-y-1">
            <li>• Tree-shaking automatique avec Next.js</li>
            <li>• Code splitting par route</li>
            <li>• Compression gzip/brotli</li>
            <li>• Lazy loading des composants non critiques</li>
            <li>• Minification et optimisation du bundle</li>
          </ul>
        </div>
      </CardContent>
    </Card>
  )
}

export { BundleAnalyzer }
