"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { ProfileDetector, type UserProfile } from "@/lib/chatbot-profile-detector"
import { getNextStep, getWelcomeMessage } from "@/lib/chatbot-conversation-flows"
import { searchKnowledge } from "@/lib/chatbot-knowledge-base"

interface TestScenario {
  name: string
  profile: UserProfile
  userInputs: string[]
  expectedFlow: string[]
}

const testScenarios: TestScenario[] = [
  {
    name: "Primo-accédant - Flux complet",
    profile: "acheteur_primo",
    userInputs: [
      "Je veux acheter mon premier appartement",
      "Oui, premier achat",
      "500-800k€",
      "Paris intra-muros",
      "3 pièces",
      "Dans les 6 mois",
    ],
    expectedFlow: ["welcome", "budget", "zone", "property_type", "timeline"],
  },
  {
    name: "Locataire - Recherche logement",
    profile: "locataire",
    userInputs: ["Je cherche un appartement à louer", "1000€ à 1500€", "Paris est", "2 pièces", "Dans le mois"],
    expectedFlow: ["welcome", "zone", "property_type", "timeline"],
  },
  {
    name: "Vendeur - Estimation bien",
    profile: "vendeur",
    userInputs: [
      "Je veux vendre mon appartement",
      "Paris intra-muros",
      "Appartement",
      "50-80m²",
      "Délai normal (6 mois)",
    ],
    expectedFlow: ["welcome", "property_details", "surface", "timeline"],
  },
  {
    name: "Investisseur - Projet locatif",
    profile: "investisseur",
    userInputs: ["Je veux investir dans l'immobilier", "Revenus locatifs", "400-600k€", "Paris", "Appartement ancien"],
    expectedFlow: ["welcome", "budget", "zone", "property_type"],
  },
]

export default function ChatbotTestSimulator() {
  const [currentTest, setCurrentTest] = useState<TestScenario | null>(null)
  const [testResults, setTestResults] = useState<Record<string, any>>({})
  const [isRunning, setIsRunning] = useState(false)

  const runTest = async (scenario: TestScenario) => {
    setIsRunning(true)
    setCurrentTest(scenario)

    const detector = new ProfileDetector()
    const results = {
      scenario: scenario.name,
      steps: [],
      profileDetection: [],
      qualificationData: {},
      success: true,
      errors: [],
      knowledgeResults: 0,
    }

    try {
      let currentStep = "welcome"
      let currentProfile: UserProfile = "unknown"

      // Test de détection de profil
      for (const input of scenario.userInputs) {
        const detectedProfile = detector.analyzeMessage(input)
        results.profileDetection.push({
          input,
          detectedProfile,
          confidence: detector.getContext().confidence,
        })

        if (detectedProfile !== "unknown") {
          currentProfile = detectedProfile
        }
      }

      // Test du flux de conversation
      if (currentProfile !== "unknown") {
        const welcomeStep = getWelcomeMessage(currentProfile)
        results.steps.push({
          stepId: welcomeStep.id,
          message: welcomeStep.message,
          options: welcomeStep.options,
        })

        currentStep = welcomeStep.id

        // Simuler la progression dans le flux
        for (let i = 1; i < scenario.userInputs.length; i++) {
          const userChoice = scenario.userInputs[i]
          const nextStep = getNextStep(currentProfile, currentStep, userChoice)

          if (nextStep) {
            results.steps.push({
              stepId: nextStep.id,
              message: nextStep.message,
              options: nextStep.options,
              qualification: nextStep.qualification,
              action: nextStep.action,
            })

            // Sauvegarder les données de qualification
            if (nextStep.qualification) {
              results.qualificationData[nextStep.qualification] = userChoice
            }

            currentStep = nextStep.id
          } else {
            results.errors.push(`Pas d'étape suivante trouvée pour: ${userChoice}`)
          }
        }
      } else {
        results.errors.push("Profil non détecté")
        results.success = false
      }

      // Test de la base de connaissances
      const knowledgeTest = searchKnowledge(scenario.userInputs[0])
      results.knowledgeResults = knowledgeTest.length

      setTestResults((prev) => ({
        ...prev,
        [scenario.name]: results,
      }))
    } catch (error) {
      results.errors.push(`Erreur: ${error}`)
      results.success = false
    }

    setIsRunning(false)
  }

  const runAllTests = async () => {
    setTestResults({})
    for (const scenario of testScenarios) {
      await runTest(scenario)
      // Petite pause entre les tests
      await new Promise((resolve) => setTimeout(resolve, 500))
    }
  }

  const successfulTests = Object.values(testResults).filter((r: any) => r.success).length
  const totalTests = Object.keys(testResults).length

  return (
    <div className="p-6 space-y-6 max-w-7xl mx-auto">
      {/* Header */}
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Test Automatisé du Chatbot</h1>
            <p className="text-gray-600 mt-2">
              Validation complète des flux de conversation et de la détection de profil
            </p>
          </div>
          <div className="text-right">
            <Button
              onClick={runAllTests}
              disabled={isRunning}
              className="bg-[#002395] hover:bg-[#001875] text-white px-6 py-3"
            >
              {isRunning ? "Tests en cours..." : "🚀 Lancer tous les tests"}
            </Button>
            {totalTests > 0 && (
              <div className="mt-2 text-sm">
                <span className="text-green-600 font-medium">{successfulTests}</span>
                <span className="text-gray-500">/{totalTests} tests réussis</span>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Scénarios de test */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {testScenarios.map((scenario) => (
          <div key={scenario.name} className="bg-white rounded-lg shadow-sm border">
            <div className="p-4 border-b bg-gray-50 rounded-t-lg">
              <h3 className="text-lg font-semibold text-gray-900">{scenario.name}</h3>
              <p className="text-sm text-gray-600">
                Profil cible: <span className="font-medium">{scenario.profile}</span>
              </p>
            </div>
            <div className="p-4">
              <div className="space-y-3">
                <h4 className="font-medium text-gray-900">Inputs utilisateur simulés:</h4>
                <div className="space-y-2">
                  {scenario.userInputs.map((input, index) => (
                    <div key={index} className="bg-blue-50 border border-blue-200 p-3 rounded-lg">
                      <span className="text-xs font-medium text-blue-600">Étape {index + 1}</span>
                      <div className="text-sm text-gray-800 mt-1">"{input}"</div>
                    </div>
                  ))}
                </div>
                <Button
                  size="sm"
                  onClick={() => runTest(scenario)}
                  disabled={isRunning}
                  className="w-full mt-4 bg-blue-600 hover:bg-blue-700"
                >
                  🧪 Tester ce scénario
                </Button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Résultats des tests */}
      {Object.keys(testResults).length > 0 && (
        <div className="space-y-6">
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <h2 className="text-2xl font-bold text-gray-900 mb-4">📊 Résultats des Tests</h2>

            {/* Statistiques globales */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
              <div className="bg-green-50 border border-green-200 rounded-lg p-4 text-center">
                <div className="text-2xl font-bold text-green-600">
                  {Object.values(testResults).filter((r: any) => r.success).length}
                </div>
                <div className="text-sm text-green-700">Tests réussis</div>
              </div>
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 text-center">
                <div className="text-2xl font-bold text-red-600">
                  {Object.values(testResults).filter((r: any) => !r.success).length}
                </div>
                <div className="text-sm text-red-700">Tests échoués</div>
              </div>
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 text-center">
                <div className="text-2xl font-bold text-blue-600">
                  {Object.values(testResults).reduce((acc: number, r: any) => acc + r.steps.length, 0)}
                </div>
                <div className="text-sm text-blue-700">Étapes testées</div>
              </div>
              <div className="bg-purple-50 border border-purple-200 rounded-lg p-4 text-center">
                <div className="text-2xl font-bold text-purple-600">
                  {Object.values(testResults).reduce(
                    (acc: number, r: any) => acc + Object.keys(r.qualificationData).length,
                    0,
                  )}
                </div>
                <div className="text-sm text-purple-700">Données qualifiées</div>
              </div>
            </div>
          </div>

          {/* Détails des résultats */}
          {Object.entries(testResults).map(([testName, results]: [string, any]) => (
            <div
              key={testName}
              className={`bg-white rounded-lg shadow-sm border-l-4 ${results.success ? "border-l-green-500" : "border-l-red-500"}`}
            >
              <div className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-xl font-semibold text-gray-900">{testName}</h3>
                  <span
                    className={`px-3 py-1 rounded-full text-sm font-medium ${results.success ? "bg-green-100 text-green-800" : "bg-red-100 text-red-800"}`}
                  >
                    {results.success ? "✅ Succès" : "❌ Échec"}
                  </span>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  {/* Détection de profil */}
                  <div>
                    <h4 className="font-medium text-gray-900 mb-3">🎯 Détection de Profil</h4>
                    <div className="space-y-2">
                      {results.profileDetection.map((detection: any, index: number) => (
                        <div key={index} className="bg-gray-50 border rounded-lg p-3">
                          <div className="text-sm">
                            <div className="font-medium text-gray-700">Input:</div>
                            <div className="text-gray-600 mb-2">"{detection.input}"</div>
                            <div className="flex justify-between items-center">
                              <span className="text-sm">
                                <strong>Profil:</strong> {detection.detectedProfile}
                              </span>
                              <span className="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded">
                                {Math.round(detection.confidence * 100)}%
                              </span>
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>

                  {/* Flux de conversation */}
                  <div>
                    <h4 className="font-medium text-gray-900 mb-3">💬 Flux de Conversation</h4>
                    <div className="space-y-2">
                      {results.steps.map((step: any, index: number) => (
                        <div key={index} className="border-l-4 border-blue-500 bg-blue-50 pl-4 py-2">
                          <div className="font-medium text-blue-900">
                            Étape {index + 1}: {step.stepId}
                          </div>
                          <div className="text-sm text-blue-700 mb-1">{step.message.substring(0, 100)}...</div>
                          {step.options && (
                            <div className="text-xs text-blue-600">
                              <strong>Options:</strong> {step.options.slice(0, 3).join(", ")}
                              {step.options.length > 3 && "..."}
                            </div>
                          )}
                          {step.qualification && (
                            <div className="text-xs text-green-600 mt-1">
                              <strong>📝 Qualification:</strong> {step.qualification}
                            </div>
                          )}
                          {step.action && (
                            <div className="text-xs text-purple-600 mt-1">
                              <strong>⚡ Action:</strong> {step.action}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  </div>
                </div>

                {/* Données de qualification */}
                {Object.keys(results.qualificationData).length > 0 && (
                  <div className="mt-6">
                    <h4 className="font-medium text-gray-900 mb-3">📋 Données Qualifiées</h4>
                    <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                        {Object.entries(results.qualificationData).map(([key, value]) => (
                          <div key={key} className="flex justify-between items-center">
                            <span className="font-medium text-green-800">{key}:</span>
                            <span className="text-green-700 bg-white px-2 py-1 rounded text-sm">{value as string}</span>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                )}

                {/* Base de connaissances */}
                <div className="mt-4">
                  <h4 className="font-medium text-gray-900 mb-2">📚 Base de Connaissances</h4>
                  <div className="text-sm text-gray-600 bg-gray-50 p-3 rounded">
                    {results.knowledgeResults} résultat(s) trouvé(s) pour la première requête
                  </div>
                </div>

                {/* Erreurs */}
                {results.errors.length > 0 && (
                  <div className="mt-4">
                    <h4 className="font-medium text-red-600 mb-2">⚠️ Erreurs Détectées</h4>
                    <div className="bg-red-50 border border-red-200 rounded-lg p-3">
                      <ul className="text-sm text-red-700 space-y-1">
                        {results.errors.map((error: string, index: number) => (
                          <li key={index}>• {error}</li>
                        ))}
                      </ul>
                    </div>
                  </div>
                )}
              </div>
            </div>
          ))}
        </div>
      )}

      {/* Instructions */}
      {Object.keys(testResults).length === 0 && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-2">🚀 Comment utiliser ce testeur</h3>
          <ul className="text-blue-800 space-y-2">
            <li>• Cliquez sur "Lancer tous les tests" pour valider automatiquement tous les scénarios</li>
            <li>• Ou testez individuellement chaque scénario avec le bouton "Tester ce scénario"</li>
            <li>• Les résultats montrent la détection de profil, le flux de conversation et les données qualifiées</li>
            <li>• Un test est réussi si le profil est détecté et le flux se déroule sans erreur</li>
          </ul>
        </div>
      )}
    </div>
  )
}
