"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Info } from "lucide-react"
import Link from "next/link"

type ServiceLevel = "essentiel" | "premium" | "serenite"
type PropertyType = "apartment" | "house" | "commercial"

export default function FeeCalculator() {
  const [rent, setRent] = useState<number>(1500)
  const [propertyType, setPropertyType] = useState<PropertyType>("apartment")
  const [serviceLevel, setServiceLevel] = useState<ServiceLevel>("essentiel")
  const [monthlyFee, setMonthlyFee] = useState<number>(0)
  const [yearlyFee, setYearlyFee] = useState<number>(0)
  const [showTooltip, setShowTooltip] = useState<string | null>(null)

  // Taux fixes par niveau de service
  const rates = {
    essentiel: 2, // 2% HT pour le niveau Essentiel
    premium: 3.9, // 3.9% HT pour le niveau Premium
    serenite: 6.9, // 6.9% HT pour le niveau Sérénité
  }

  // Descriptions des niveaux de service
  const serviceDescriptions = {
    essentiel: "La solution idéale pour les propriétaires qui souhaitent une gestion simple et efficace.",
    premium: "Une solution complète pour une tranquillité d'esprit totale.",
    serenite: "La solution tout inclus pour une gestion sans souci.",
  }

  // Calculer les frais en fonction des paramètres
  useEffect(() => {
    // Utiliser le taux fixe selon le niveau de service
    const rate = rates[serviceLevel]

    // Calcul final
    const monthly = (rent * rate) / 100
    const yearly = monthly * 12

    setMonthlyFee(monthly)
    setYearlyFee(yearly)
  }, [rent, serviceLevel, rates])

  return (
    <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
      <h3 className="text-xl font-bold mb-6 text-[#002395]">Calculateur de frais de gestion</h3>

      <div className="space-y-6">
        <div>
          <label htmlFor="rent-amount" className="block text-base font-medium mb-2">
            Montant du loyer mensuel (€)
          </label>
          <div className="flex items-center gap-4">
            <input
              type="range"
              id="rent-slider"
              min={500}
              max={5000}
              step={50}
              value={rent}
              onChange={(e) => setRent(Number(e.target.value))}
              className="flex-1 h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer accent-[#002395]"
            />
            <input
              id="rent-amount"
              type="number"
              value={rent}
              onChange={(e) => setRent(Number(e.target.value))}
              className="w-32 px-3 py-2 border border-gray-300 rounded-md text-right"
            />
          </div>
        </div>

        <div>
          <label htmlFor="property-type" className="block text-base font-medium mb-2">
            Type de bien
          </label>
          <select
            id="property-type"
            value={propertyType}
            onChange={(e) => setPropertyType(e.target.value as PropertyType)}
            className="w-full h-12 rounded-md border border-gray-300 bg-white px-3 py-2 text-base"
          >
            <option value="apartment">Appartement</option>
            <option value="house">Maison</option>
            <option value="commercial">Local commercial</option>
          </select>
        </div>

        <div>
          <div className="flex items-center gap-2 mb-2">
            <label className="block text-base font-medium">Niveau de service</label>
            <div className="relative">
              <button
                type="button"
                onClick={() => setShowTooltip(showTooltip ? null : "services")}
                className="text-gray-500 hover:text-[#002395]"
              >
                <Info size={16} />
              </button>
              {showTooltip === "services" && (
                <div className="absolute z-10 bg-white p-4 rounded-md shadow-lg border border-gray-200 w-64 right-0 mt-2">
                  <h4 className="font-semibold mb-2">Niveaux de service</h4>
                  <ul className="space-y-2 text-sm">
                    <li>
                      <span className="font-medium">Essentiel :</span> Gestion administrative et financière de base.
                    </li>
                    <li>
                      <span className="font-medium">Premium :</span> Inclut la gestion technique et le suivi des
                      travaux.
                    </li>
                    <li>
                      <span className="font-medium">Sérénité :</span> Tout inclus avec garantie loyers impayés et
                      assurances.
                    </li>
                  </ul>
                  <button className="text-xs text-[#002395] mt-2" onClick={() => setShowTooltip(null)}>
                    Fermer
                  </button>
                </div>
              )}
            </div>
          </div>
          <div className="grid grid-cols-3 gap-3">
            <button
              className={`p-3 border rounded-md text-center transition-colors ${
                serviceLevel === "essentiel"
                  ? "bg-[#002395] text-white border-[#002395]"
                  : "hover:bg-gray-100 border-gray-300"
              }`}
              onClick={() => setServiceLevel("essentiel")}
            >
              Essentiel
            </button>
            <button
              className={`p-3 border rounded-md text-center transition-colors ${
                serviceLevel === "premium"
                  ? "bg-[#002395] text-white border-[#002395]"
                  : "hover:bg-gray-100 border-gray-300"
              }`}
              onClick={() => setServiceLevel("premium")}
            >
              Premium
            </button>
            <button
              className={`p-3 border rounded-md text-center transition-colors ${
                serviceLevel === "serenite"
                  ? "bg-[#002395] text-white border-[#002395]"
                  : "hover:bg-gray-100 border-gray-300"
              }`}
              onClick={() => setServiceLevel("serenite")}
            >
              Sérénité
            </button>
          </div>
          <p className="text-sm text-gray-500 mt-2 italic">{serviceDescriptions[serviceLevel]}</p>
        </div>

        <div className="bg-gray-50 p-6 rounded-lg">
          <div className="text-center mb-4">
            <div className="text-sm text-gray-500">Taux appliqué</div>
            <div className="text-3xl font-bold text-[#002395]">{rates[serviceLevel].toFixed(1)}%</div>
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="text-center">
              <div className="text-sm text-gray-500">Frais mensuels</div>
              <div className="text-2xl font-bold">{monthlyFee.toFixed(2)}€</div>
            </div>
            <div className="text-center">
              <div className="text-sm text-gray-500">Frais annuels</div>
              <div className="text-2xl font-bold">{yearlyFee.toFixed(2)}€</div>
            </div>
          </div>
        </div>

        <Link href="/contact">
          <Button className="w-full bg-[#002395] text-white hover:bg-[#001875] py-6 text-base">
            Demander un devis personnalisé
          </Button>
        </Link>
      </div>
    </div>
  )
}
