"use client"

import { useState } from "react"
import { CheckCircle, AlertCircle, Eye, ExternalLink } from "lucide-react"

interface AuditItem {
  id: string
  category: string
  item: string
  status: "ok" | "check" | "info"
  description: string
  action?: string
}

const auditItems: AuditItem[] = [
  // Cohérence visuelle
  {
    id: "visual-1",
    category: "Cohérence Visuelle",
    item: "Logo et branding",
    status: "ok",
    description: "Logo DOGES cohérent entre l'ancien et le nouveau site",
  },
  {
    id: "visual-2",
    category: "Cohérence Visuelle",
    item: "Couleurs et typographie",
    status: "ok",
    description: "Palette de couleurs bleue (#002395) maintenue",
  },
  {
    id: "visual-3",
    category: "Cohérence Visuelle",
    item: "Structure de navigation",
    status: "check",
    description: "Vérifier que tous les liens de navigation fonctionnent correctement",
  },

  // Contenu et services
  {
    id: "content-1",
    category: "Contenu & Services",
    item: "Services proposés",
    status: "ok",
    description: "4 services principaux identiques : Gestion locative, Transaction, GLI, PNO",
  },
  {
    id: "content-2",
    category: "Contenu & Services",
    item: "Tarifs et calculateurs",
    status: "ok",
    description: "Tarifs uniformisés : 2% / 3.9% / 6.9%",
  },
  {
    id: "content-3",
    category: "Contenu & Services",
    item: "Formulaires de contact",
    status: "check",
    description: "Vérifier que les formulaires fonctionnent et envoient bien les emails",
  },

  // Fonctionnalités
  {
    id: "func-1",
    category: "Fonctionnalités",
    item: "Bouton 'APPELEZ NOUS'",
    status: "check",
    description: "Tester le lien Ringover pour s'assurer qu'il fonctionne",
    action: "Tester le lien téléphonique",
  },
  {
    id: "func-2",
    category: "Fonctionnalités",
    item: "Chatbot intelligent",
    status: "info",
    description: "Nouvelle fonctionnalité ajoutée - à présenter au client",
  },
  {
    id: "func-3",
    category: "Fonctionnalités",
    item: "Calculateurs immobiliers",
    status: "info",
    description: "Fonctionnalités améliorées avec nouveaux tarifs",
  },

  // SEO et Performance
  {
    id: "seo-1",
    category: "SEO & Performance",
    item: "Métadonnées et référencement",
    status: "ok",
    description: "Toutes les pages optimisées SEO avec métadonnées complètes",
  },
  {
    id: "seo-2",
    category: "SEO & Performance",
    item: "Vitesse de chargement",
    status: "ok",
    description: "Images optimisées et code performant",
  },
  {
    id: "seo-3",
    category: "SEO & Performance",
    item: "Responsive design",
    status: "ok",
    description: "Site adaptatif pour tous les appareils",
  },

  // Domaine et configuration
  {
    id: "domain-1",
    category: "Domaine & Configuration",
    item: "Domaine dogesadb.fr",
    status: "ok",
    description: "Domaine confirmé par le client",
  },
  {
    id: "domain-2",
    category: "Domaine & Configuration",
    item: "Redirections www",
    status: "check",
    description: "S'assurer que www.dogesadb.fr redirige vers dogesadb.fr",
  },
  {
    id: "domain-3",
    category: "Domaine & Configuration",
    item: "Certificat SSL",
    status: "check",
    description: "Vérifier que HTTPS est bien configuré",
  },
]

export default function FinalAuditChecklist() {
  const [checkedItems, setCheckedItems] = useState<Set<string>>(new Set())

  const toggleItem = (id: string) => {
    const newChecked = new Set(checkedItems)
    if (newChecked.has(id)) {
      newChecked.delete(id)
    } else {
      newChecked.add(id)
    }
    setCheckedItems(newChecked)
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "ok":
        return <CheckCircle className="h-5 w-5 text-green-500" />
      case "check":
        return <AlertCircle className="h-5 w-5 text-yellow-500" />
      case "info":
        return <Eye className="h-5 w-5 text-blue-500" />
      default:
        return <AlertCircle className="h-5 w-5 text-gray-400" />
    }
  }

  const categories = [...new Set(auditItems.map((item) => item.category))]

  const okCount = auditItems.filter((item) => item.status === "ok").length
  const checkCount = auditItems.filter((item) => item.status === "check").length
  const infoCount = auditItems.filter((item) => item.status === "info").length

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="bg-white rounded-lg shadow-lg p-6">
        <h1 className="text-3xl font-bold mb-6 text-[#002395]">Audit Final - Site DOGES</h1>

        <div className="mb-6 p-4 bg-green-50 rounded-lg border border-green-200">
          <h2 className="text-lg font-semibold text-green-800 mb-2">✅ Domaine confirmé : dogesadb.fr</h2>
          <p className="text-green-700">Le client a confirmé le domaine définitif et les informations légales.</p>
        </div>

        {/* Résumé */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <div className="bg-green-50 p-4 rounded-lg border border-green-200">
            <div className="flex items-center">
              <CheckCircle className="h-6 w-6 text-green-500 mr-2" />
              <div>
                <div className="font-semibold text-green-800">{okCount} Validés</div>
                <div className="text-sm text-green-600">Conformes et prêts</div>
              </div>
            </div>
          </div>

          <div className="bg-yellow-50 p-4 rounded-lg border border-yellow-200">
            <div className="flex items-center">
              <AlertCircle className="h-6 w-6 text-yellow-500 mr-2" />
              <div>
                <div className="font-semibold text-yellow-800">{checkCount} À Tester</div>
                <div className="text-sm text-yellow-600">Tests fonctionnels</div>
              </div>
            </div>
          </div>

          <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
            <div className="flex items-center">
              <Eye className="h-6 w-6 text-blue-500 mr-2" />
              <div>
                <div className="font-semibold text-blue-800">{infoCount} Nouveautés</div>
                <div className="text-sm text-blue-600">À présenter au client</div>
              </div>
            </div>
          </div>
        </div>

        {/* Checklist par catégorie */}
        {categories.map((category) => (
          <div key={category} className="mb-8">
            <h2 className="text-xl font-semibold mb-4 text-gray-800 border-b pb-2">{category}</h2>
            <div className="space-y-3">
              {auditItems
                .filter((item) => item.category === category)
                .map((item) => (
                  <div
                    key={item.id}
                    className="flex items-start p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                  >
                    <div className="mr-3 mt-0.5">{getStatusIcon(item.status)}</div>
                    <div className="flex-1">
                      <div className="font-medium text-gray-900">{item.item}</div>
                      <div className="text-sm text-gray-600 mt-1">{item.description}</div>
                      {item.action && <div className="text-xs text-blue-600 mt-1 font-medium">→ {item.action}</div>}
                    </div>
                    <input
                      type="checkbox"
                      checked={checkedItems.has(item.id)}
                      onChange={() => toggleItem(item.id)}
                      className="ml-4 h-4 w-4 text-[#002395] rounded"
                    />
                  </div>
                ))}
            </div>
          </div>
        ))}

        {/* Tests à effectuer */}
        <div className="mt-8 p-6 bg-yellow-50 rounded-lg border border-yellow-200">
          <h3 className="text-lg font-semibold text-yellow-800 mb-3">🧪 Tests à effectuer avant validation</h3>
          <div className="grid md:grid-cols-2 gap-4">
            <div>
              <h4 className="font-medium text-yellow-800 mb-2">Tests fonctionnels :</h4>
              <ul className="space-y-1 text-yellow-700 text-sm">
                <li>• Tester le bouton "APPELEZ NOUS" (Ringover)</li>
                <li>• Vérifier tous les formulaires de contact</li>
                <li>• Tester les calculateurs avec les nouveaux tarifs</li>
                <li>• Vérifier la navigation mobile</li>
              </ul>
            </div>
            <div>
              <h4 className="font-medium text-yellow-800 mb-2">Tests techniques :</h4>
              <ul className="space-y-1 text-yellow-700 text-sm">
                <li>• Vérifier HTTPS et certificat SSL</li>
                <li>• Tester les redirections www</li>
                <li>• Vérifier la vitesse de chargement</li>
                <li>• Tester sur différents navigateurs</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Nouveautés à présenter */}
        <div className="mt-6 p-6 bg-blue-50 rounded-lg border border-blue-200">
          <h3 className="text-lg font-semibold text-blue-800 mb-3">🆕 Nouveautés à présenter au client</h3>
          <ul className="space-y-2 text-blue-700">
            <li>
              • <strong>Chatbot intelligent</strong> : Assistant conversationnel pour qualifier les prospects
            </li>
            <li>
              • <strong>Calculateurs améliorés</strong> : Nouveaux tarifs et redirection vers formulaire de devis
            </li>
            <li>
              • <strong>SEO optimisé</strong> : Meilleur référencement avec pages spécialisées par arrondissement
            </li>
            <li>
              • <strong>Blog enrichi</strong> : Articles de qualité pour le référencement naturel
            </li>
            <li>
              • <strong>Design responsive</strong> : Expérience optimisée sur tous les appareils
            </li>
          </ul>
        </div>

        {/* Verdict final */}
        <div className="mt-8 p-6 bg-gradient-to-r from-green-50 to-blue-50 rounded-lg border border-green-200">
          <div className="text-center">
            <h3 className="text-2xl font-bold text-green-800 mb-2">🎉 SITE PRÊT POUR VALIDATION CLIENT</h3>
            <p className="text-green-700 mb-4">
              Le site est techniquement prêt et apporte de nombreuses améliorations par rapport à l'existant.
            </p>
            <div className="flex justify-center space-x-4">
              <a
                href="https://dogesadb.fr"
                target="_blank"
                rel="noopener noreferrer"
                className="inline-flex items-center px-4 py-2 bg-[#002395] text-white rounded-lg hover:bg-[#001875] transition-colors"
              >
                <ExternalLink className="h-4 w-4 mr-2" />
                Voir le site actuel
              </a>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
