"use client"

import { useEffect, useRef, useState } from "react"
import mapboxgl from "mapbox-gl"
import "mapbox-gl/dist/mapbox-gl.css"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsList, TabsTrigger } from "@/components/ui/tabs"

// Données enrichies pour chaque arrondissement de Paris
const parisDistrictsData = [
  {
    id: "75001",
    name: "1er Arrondissement",
    center: [2.3417, 48.8625],
    vacancyRate: 3.2,
    averageOccupancyDuration: 2.8,
    tenantProfile: "Cadres supérieurs, expatriés, couples sans enfants",
    description: "Quartier historique et touristique avec une forte demande locative premium.",
  },
  {
    id: "75002",
    name: "2ème Arrondissement",
    center: [2.347, 48.8696],
    vacancyRate: 2.9,
    averageOccupancyDuration: 2.5,
    tenantProfile: "Jeunes actifs, professionnels du secteur financier",
    description: "Secteur dynamique avec une forte concentration d'entreprises et de startups.",
  },
  {
    id: "75003",
    name: "3ème Arrondissement",
    center: [2.3621, 48.8633],
    vacancyRate: 2.7,
    averageOccupancyDuration: 3.1,
    tenantProfile: "Artistes, créatifs, jeunes couples",
    description: "Partie nord du Marais, très prisée pour son ambiance branchée et culturelle.",
  },
  {
    id: "75004",
    name: "4ème Arrondissement",
    center: [2.3548, 48.8566],
    vacancyRate: 2.5,
    averageOccupancyDuration: 3.3,
    tenantProfile: "Cadres, familles aisées, touristes (locations saisonnières)",
    description: "Cœur historique de Paris avec une forte attractivité touristique et résidentielle.",
  },
  {
    id: "75005",
    name: "5ème Arrondissement",
    center: [2.3508, 48.8448],
    vacancyRate: 2.3,
    averageOccupancyDuration: 4.2,
    tenantProfile: "Étudiants, enseignants, chercheurs, familles",
    description: "Quartier Latin avec une forte présence universitaire et intellectuelle.",
  },
  {
    id: "75006",
    name: "6ème Arrondissement",
    center: [2.335, 48.8493],
    vacancyRate: 2.1,
    averageOccupancyDuration: 4.5,
    tenantProfile: "Cadres supérieurs, professions libérales, familles aisées",
    description: "Quartier prestigieux avec une clientèle haut de gamme et stable.",
  },
  {
    id: "75007",
    name: "7ème Arrondissement",
    center: [2.3174, 48.857],
    vacancyRate: 2.0,
    averageOccupancyDuration: 5.1,
    tenantProfile: "Diplomates, hauts fonctionnaires, familles très aisées",
    description: "Secteur résidentiel de prestige avec une clientèle internationale.",
  },
  {
    id: "75008",
    name: "8ème Arrondissement",
    center: [2.3125, 48.8728],
    vacancyRate: 3.5,
    averageOccupancyDuration: 3.2,
    tenantProfile: "Cadres dirigeants, expatriés, investisseurs étrangers",
    description: "Mélange de bureaux et résidences haut de gamme avec forte demande internationale.",
  },
  {
    id: "75009",
    name: "9ème Arrondissement",
    center: [2.337, 48.8765],
    vacancyRate: 3.1,
    averageOccupancyDuration: 2.9,
    tenantProfile: "Jeunes actifs, couples, professionnels du digital",
    description: "Quartier en pleine gentrification avec une population jeune et dynamique.",
  },
  {
    id: "75010",
    name: "10ème Arrondissement",
    center: [2.3568, 48.8762],
    vacancyRate: 3.8,
    averageOccupancyDuration: 2.6,
    tenantProfile: "Jeunes actifs, artistes, population multiculturelle",
    description: "Secteur en transformation avec une forte mixité sociale et culturelle.",
  },
  {
    id: "75011",
    name: "11ème Arrondissement",
    center: [2.3795, 48.8583],
    vacancyRate: 3.3,
    averageOccupancyDuration: 2.7,
    tenantProfile: "Jeunes actifs, créatifs, familles",
    description: "Arrondissement dynamique et branché avec une forte demande locative.",
  },
  {
    id: "75012",
    name: "12ème Arrondissement",
    center: [2.3883, 48.8402],
    vacancyRate: 3.0,
    averageOccupancyDuration: 3.5,
    tenantProfile: "Familles, jeunes couples, fonctionnaires",
    description: "Quartier résidentiel équilibré avec une bonne qualité de vie.",
  },
  {
    id: "75013",
    name: "13ème Arrondissement",
    center: [2.3561, 48.8322],
    vacancyRate: 3.2,
    averageOccupancyDuration: 3.8,
    tenantProfile: "Étudiants, chercheurs, familles, population asiatique",
    description: "Secteur diversifié avec une forte présence universitaire et multiculturelle.",
  },
  {
    id: "75014",
    name: "14ème Arrondissement",
    center: [2.3264, 48.833],
    vacancyRate: 2.8,
    averageOccupancyDuration: 4.0,
    tenantProfile: "Familles, enseignants, professions intellectuelles",
    description: "Quartier résidentiel calme avec une population stable.",
  },
  {
    id: "75015",
    name: "15ème Arrondissement",
    center: [2.3002, 48.8417],
    vacancyRate: 2.6,
    averageOccupancyDuration: 4.3,
    tenantProfile: "Familles, cadres moyens et supérieurs, fonctionnaires",
    description: "Plus grand arrondissement de Paris avec une clientèle familiale fidèle.",
  },
  {
    id: "75016",
    name: "16ème Arrondissement",
    center: [2.2769, 48.8637],
    vacancyRate: 2.4,
    averageOccupancyDuration: 5.2,
    tenantProfile: "Familles aisées, diplomates, cadres supérieurs",
    description: "Quartier résidentiel haut de gamme avec une clientèle stable et fortunée.",
  },
  {
    id: "75017",
    name: "17ème Arrondissement",
    center: [2.3141, 48.8835],
    vacancyRate: 2.9,
    averageOccupancyDuration: 3.7,
    tenantProfile: "Familles, cadres, professions libérales",
    description: "Arrondissement contrasté entre zones très huppées et secteurs plus accessibles.",
  },
  {
    id: "75018",
    name: "18ème Arrondissement",
    center: [2.3488, 48.8925],
    vacancyRate: 4.1,
    averageOccupancyDuration: 2.4,
    tenantProfile: "Artistes, jeunes actifs, population multiculturelle",
    description: "Secteur très contrasté entre Montmartre touristique et quartiers populaires.",
  },
  {
    id: "75019",
    name: "19ème Arrondissement",
    center: [2.3822, 48.8849],
    vacancyRate: 4.3,
    averageOccupancyDuration: 2.8,
    tenantProfile: "Familles, jeunes actifs, population diversifiée",
    description: "Arrondissement en pleine mutation avec un fort potentiel de valorisation.",
  },
  {
    id: "75020",
    name: "20ème Arrondissement",
    center: [2.3984, 48.865],
    vacancyRate: 3.9,
    averageOccupancyDuration: 3.0,
    tenantProfile: "Jeunes actifs, artistes, familles, population diversifiée",
    description: "Quartier populaire en gentrification progressive avec des loyers encore accessibles.",
  },
]

type DataCategory = "vacancyRate" | "averageOccupancyDuration" | "tenantProfile"

interface LocalDataMapProps {
  className?: string
  initialDistrict?: string
  mapboxToken: string
}

export default function LocalDataMap({ className, initialDistrict, mapboxToken }: LocalDataMapProps) {
  const mapContainer = useRef<HTMLDivElement>(null)
  const map = useRef<mapboxgl.Map | null>(null)
  const [selectedDistrict, setSelectedDistrict] = useState<string | null>(initialDistrict || null)
  const [activeTab, setActiveTab] = useState<DataCategory>("vacancyRate")
  const [mapLoaded, setMapLoaded] = useState(false)

  // Fonction pour obtenir la couleur en fonction du taux de vacance
  const getVacancyRateColor = (rate: number) => {
    if (rate < 2.5) return "#4ade80" // vert - très faible vacance
    if (rate < 3.0) return "#a3e635" // vert clair - faible vacance
    if (rate < 3.5) return "#facc15" // jaune - vacance moyenne
    if (rate < 4.0) return "#fb923c" // orange - vacance élevée
    return "#ef4444" // rouge - vacance très élevée
  }

  // Fonction pour obtenir la couleur en fonction de la durée d'occupation
  const getOccupancyDurationColor = (duration: number) => {
    if (duration > 4.5) return "#4ade80" // vert - très longue durée
    if (duration > 3.5) return "#a3e635" // vert clair - longue durée
    if (duration > 2.5) return "#facc15" // jaune - durée moyenne
    if (duration > 2.0) return "#fb923c" // orange - courte durée
    return "#ef4444" // rouge - très courte durée
  }

  // Initialisation de la carte
  useEffect(() => {
    if (!mapContainer.current || map.current || !mapboxToken) return

    // Configuration du token Mapbox
    mapboxgl.accessToken = mapboxToken

    map.current = new mapboxgl.Map({
      container: mapContainer.current,
      style: "mapbox://styles/mapbox/light-v11",
      center: [2.3522, 48.8566], // Centre de Paris
      zoom: 11.5,
      minZoom: 10,
      maxZoom: 15,
    })

    map.current.on("load", () => {
      setMapLoaded(true)
    })

    return () => {
      map.current?.remove()
      map.current = null
    }
  }, [mapboxToken])

  // Ajout des marqueurs et popups pour chaque arrondissement
  useEffect(() => {
    if (!map.current || !mapLoaded) return

    // Supprimer les marqueurs existants
    const markers = document.querySelectorAll(".district-marker")
    markers.forEach((marker) => marker.remove())

    // Ajouter les nouveaux marqueurs
    parisDistrictsData.forEach((district) => {
      // Créer un élément div pour le marqueur
      const markerElement = document.createElement("div")
      markerElement.className = "district-marker"
      markerElement.style.width = "30px"
      markerElement.style.height = "30px"
      markerElement.style.borderRadius = "50%"
      markerElement.style.display = "flex"
      markerElement.style.alignItems = "center"
      markerElement.style.justifyContent = "center"
      markerElement.style.fontWeight = "bold"
      markerElement.style.fontSize = "12px"
      markerElement.style.cursor = "pointer"
      markerElement.style.transition = "all 0.3s ease"

      // Définir la couleur en fonction de l'onglet actif
      if (activeTab === "vacancyRate") {
        markerElement.style.backgroundColor = getVacancyRateColor(district.vacancyRate)
        markerElement.style.color = district.vacancyRate > 3.5 ? "white" : "black"
        markerElement.textContent = `${district.vacancyRate}%`
      } else if (activeTab === "averageOccupancyDuration") {
        markerElement.style.backgroundColor = getOccupancyDurationColor(district.averageOccupancyDuration)
        markerElement.style.color = district.averageOccupancyDuration < 2.5 ? "white" : "black"
        markerElement.textContent = `${district.averageOccupancyDuration}`
      } else {
        markerElement.style.backgroundColor = "#6366f1"
        markerElement.style.color = "white"
        markerElement.textContent = district.id.substring(3)
      }

      // Ajouter un effet de survol
      markerElement.onmouseover = () => {
        markerElement.style.transform = "scale(1.2)"
        markerElement.style.zIndex = "10"
      }
      markerElement.onmouseout = () => {
        markerElement.style.transform = "scale(1)"
        markerElement.style.zIndex = "1"
      }

      // Ajouter un gestionnaire de clic
      markerElement.onclick = () => {
        setSelectedDistrict(district.id)
      }

      // Ajouter le marqueur à la carte
      new mapboxgl.Marker({ element: markerElement }).setLngLat(district.center).addTo(map.current!)
    })
  }, [mapLoaded, activeTab])

  // Trouver les données du district sélectionné
  const selectedDistrictData = selectedDistrict
    ? parisDistrictsData.find((district) => district.id === selectedDistrict)
    : null

  if (!mapboxToken) {
    return (
      <Card className="flex-1">
        <CardContent className="flex items-center justify-center h-64">
          <p className="text-muted-foreground">Configuration de la carte en cours...</p>
        </CardContent>
      </Card>
    )
  }

  return (
    <div className={`flex flex-col lg:flex-row gap-6 ${className}`}>
      <Card className="flex-1">
        <CardHeader>
          <CardTitle>Carte des données locales par arrondissement</CardTitle>
          <CardDescription>Explorez les données locatives par arrondissement de Paris</CardDescription>
        </CardHeader>
        <CardContent>
          <Tabs value={activeTab} onValueChange={(value) => setActiveTab(value as DataCategory)}>
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="vacancyRate">Taux de vacance</TabsTrigger>
              <TabsTrigger value="averageOccupancyDuration">Durée d&apos;occupation</TabsTrigger>
              <TabsTrigger value="tenantProfile">Profil locataires</TabsTrigger>
            </TabsList>
          </Tabs>

          <div ref={mapContainer} className="w-full h-[400px] mt-4 rounded-md overflow-hidden" />

          <div className="mt-4 flex justify-center gap-4 flex-wrap">
            {activeTab === "vacancyRate" && (
              <div className="flex items-center gap-6 text-sm">
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#4ade80" }}></div>
                  <span>&lt; 2.5%</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#a3e635" }}></div>
                  <span>2.5-3.0%</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#facc15" }}></div>
                  <span>3.0-3.5%</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#fb923c" }}></div>
                  <span>3.5-4.0%</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#ef4444" }}></div>
                  <span>&gt; 4.0%</span>
                </div>
              </div>
            )}

            {activeTab === "averageOccupancyDuration" && (
              <div className="flex items-center gap-6 text-sm">
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#4ade80" }}></div>
                  <span>&gt; 4.5 ans</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#a3e635" }}></div>
                  <span>3.5-4.5 ans</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#facc15" }}></div>
                  <span>2.5-3.5 ans</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#fb923c" }}></div>
                  <span>2.0-2.5 ans</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 rounded-full" style={{ backgroundColor: "#ef4444" }}></div>
                  <span>&lt; 2.0 ans</span>
                </div>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      <Card className="w-full lg:w-96">
        <CardHeader>
          <CardTitle>{selectedDistrictData ? selectedDistrictData.name : "Sélectionnez un arrondissement"}</CardTitle>
          <CardDescription>
            {selectedDistrictData ? "Données locatives détaillées" : "Cliquez sur un marqueur pour voir les détails"}
          </CardDescription>
        </CardHeader>
        <CardContent>
          {selectedDistrictData ? (
            <div className="space-y-4">
              <div>
                <h3 className="font-medium text-sm text-gray-500">Taux de vacance locative</h3>
                <p className="text-lg font-semibold">{selectedDistrictData.vacancyRate}%</p>
              </div>

              <div>
                <h3 className="font-medium text-sm text-gray-500">Durée moyenne d&apos;occupation</h3>
                <p className="text-lg font-semibold">{selectedDistrictData.averageOccupancyDuration} ans</p>
              </div>

              <div>
                <h3 className="font-medium text-sm text-gray-500">Profil type des locataires</h3>
                <p className="text-base">{selectedDistrictData.tenantProfile}</p>
              </div>

              <div>
                <h3 className="font-medium text-sm text-gray-500">Caractéristiques du marché</h3>
                <p className="text-base">{selectedDistrictData.description}</p>
              </div>

              <div className="pt-2">
                <h3 className="font-medium text-sm text-gray-500">Analyse</h3>
                <p className="text-base">
                  {selectedDistrictData.vacancyRate < 3.0 && selectedDistrictData.averageOccupancyDuration > 3.5
                    ? "Marché locatif très stable avec forte demande et faible rotation. Idéal pour investissement long terme."
                    : selectedDistrictData.vacancyRate < 3.5 && selectedDistrictData.averageOccupancyDuration > 2.5
                      ? "Bon équilibre entre stabilité locative et rotation. Investissement sécurisé avec rendement régulier."
                      : "Marché plus volatil avec rotation plus fréquente. Potentiel de rendement plus élevé mais gestion plus active nécessaire."}
                </p>
              </div>
            </div>
          ) : (
            <div className="flex flex-col items-center justify-center h-64 text-center text-gray-500">
              <p>Cliquez sur un arrondissement sur la carte pour afficher ses données locatives détaillées</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  )
}
