"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Slider } from "@/components/ui/slider"
import { Info } from "lucide-react"

export default function ManagementFeeCalculator() {
  const [rent, setRent] = useState<number>(1500)
  const [propertyType, setPropertyType] = useState<string>("apartment")
  const [serviceLevel, setServiceLevel] = useState<string>("basic")
  const [managementFee, setManagementFee] = useState<number>(0)
  const [yearlyFee, setYearlyFee] = useState<number>(0)
  const [percentageRate, setPercentageRate] = useState<number>(3.9)
  const [description, setDescription] = useState<string>(
    "La solution idéale pour les propriétaires qui souhaitent une gestion simple et efficace.",
  )

  // Définir les taux fixes par niveau de service
  const serviceLevelRates = {
    basic: 2.0,
    standard: 3.9,
    premium: 6.9,
  }

  // Définir les descriptions par niveau de service
  const serviceLevelDescriptions = {
    basic: "La solution idéale pour les propriétaires qui souhaitent une gestion simple et efficace.",
    standard: "Notre formule la plus populaire avec des services complets pour une gestion sans souci.",
    premium: "Une gestion tout inclus pour une tranquillité totale et une valorisation optimale.",
  }

  // Calculer les frais en fonction des paramètres
  useEffect(() => {
    // Utiliser le taux fixe selon le niveau de service
    const rate = serviceLevelRates[serviceLevel as keyof typeof serviceLevelRates] / 100

    // Calcul final
    const monthlyFee = rent * rate
    const yearly = monthlyFee * 12

    setPercentageRate(serviceLevelRates[serviceLevel as keyof typeof serviceLevelRates])
    setManagementFee(monthlyFee)
    setYearlyFee(yearly)
    setDescription(serviceLevelDescriptions[serviceLevel as keyof typeof serviceLevelDescriptions])
  }, [rent, propertyType, serviceLevel])

  return (
    <div className="space-y-6">
      <div>
        <Label htmlFor="rent-amount">Montant du loyer mensuel (€)</Label>
        <div className="flex items-center gap-4 mt-2">
          <Slider
            id="rent-slider"
            min={500}
            max={5000}
            step={50}
            value={[rent]}
            onValueChange={(value) => setRent(value[0])}
            className="flex-1 [&>[role=slider]]:bg-[#002395] [&>span]:bg-gray-200"
          />
          <Input
            id="rent-amount"
            type="number"
            value={rent}
            onChange={(e) => setRent(Number(e.target.value))}
            className="w-24"
          />
        </div>
      </div>

      <div>
        <Label htmlFor="property-type">Type de bien</Label>
        <select
          id="property-type"
          value={propertyType}
          onChange={(e) => setPropertyType(e.target.value)}
          className="w-full h-10 rounded-md border border-input bg-transparent px-3 py-1 text-sm shadow-sm transition-colors mt-2"
        >
          <option value="apartment">Appartement</option>
          <option value="house">Maison</option>
          <option value="commercial">Local commercial</option>
        </select>
      </div>

      <div>
        <div className="flex items-center gap-2">
          <Label htmlFor="service-level">Niveau de service</Label>
          <div
            className="text-gray-400 cursor-help"
            title="Choisissez le niveau de service qui correspond à vos besoins"
          >
            <Info className="h-4 w-4" />
          </div>
        </div>
        <div className="grid grid-cols-3 gap-2 mt-2">
          <button
            className={`p-3 border rounded-md text-center transition-colors ${
              serviceLevel === "basic" ? "bg-[#002395] text-white" : "hover:bg-gray-100"
            }`}
            onClick={() => setServiceLevel("basic")}
          >
            Essentiel
          </button>
          <button
            className={`p-3 border rounded-md text-center transition-colors ${
              serviceLevel === "standard" ? "bg-[#002395] text-white" : "hover:bg-gray-100"
            }`}
            onClick={() => setServiceLevel("standard")}
          >
            Premium
          </button>
          <button
            className={`p-3 border rounded-md text-center transition-colors ${
              serviceLevel === "premium" ? "bg-[#002395] text-white" : "hover:bg-gray-100"
            }`}
            onClick={() => setServiceLevel("premium")}
          >
            Sérénité
          </button>
        </div>
        <p className="text-sm text-gray-500 italic mt-2">{description}</p>
      </div>

      <div className="mt-4 mb-2">
        <p className="text-sm text-gray-600 italic">{description}</p>
      </div>

      <div className="bg-blue-50 p-4 rounded-lg border border-blue-100 mt-6">
        <div className="text-center mb-4">
          <div className="text-sm text-gray-600 mb-1">Taux appliqué</div>
          <div className="text-2xl font-bold text-[#002395]">{percentageRate.toFixed(1)}%</div>
        </div>

        <div className="grid grid-cols-2 gap-4 mb-4">
          <div className="text-center">
            <div className="text-sm text-gray-600 mb-1">Frais mensuels</div>
            <div className="text-xl font-bold text-[#002395]">{managementFee.toFixed(2)}€</div>
          </div>
          <div className="text-center">
            <div className="text-sm text-gray-600 mb-1">Frais annuels</div>
            <div className="text-xl font-bold text-[#002395]">{yearlyFee.toFixed(2)}€</div>
          </div>
        </div>

        <div className="text-center">
          <Button className="w-full bg-[#002395] hover:bg-[#001875] text-white">Demander un devis personnalisé</Button>
        </div>
      </div>
    </div>
  )
}
