"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Label } from "@/components/ui/label"
import { Slider } from "@/components/ui/slider"
import Link from "next/link"

export default function MortgageCalculator() {
  const [loanAmount, setLoanAmount] = useState(300000)
  const [interestRate, setInterestRate] = useState(3.5)
  const [loanTerm, setLoanTerm] = useState(25)
  const [monthlyPayment, setMonthlyPayment] = useState(0)

  useEffect(() => {
    calculateMonthlyPayment()
  }, [loanAmount, interestRate, loanTerm])

  const calculateMonthlyPayment = () => {
    // Convertir le taux d'intérêt annuel en taux mensuel
    const monthlyRate = interestRate / 100 / 12
    // Convertir la durée du prêt en mois
    const termInMonths = loanTerm * 12
    // Calculer le paiement mensuel
    if (monthlyRate === 0) {
      setMonthlyPayment(loanAmount / termInMonths)
    } else {
      const payment =
        (loanAmount * monthlyRate * Math.pow(1 + monthlyRate, termInMonths)) /
        (Math.pow(1 + monthlyRate, termInMonths) - 1)
      setMonthlyPayment(payment)
    }
  }

  const formatCurrency = (value) => {
    return new Intl.NumberFormat("fr-FR", {
      style: "currency",
      currency: "EUR",
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(value)
  }

  const handleLoanAmountChange = (value) => {
    setLoanAmount(Number.parseInt(value))
  }

  const handleInterestRateChange = (value) => {
    setInterestRate(Number.parseFloat(value))
  }

  const handleLoanTermChange = (value) => {
    setLoanTerm(Number.parseInt(value))
  }

  return (
    <Card className="bg-white rounded-xl shadow-lg border border-gray-100">
      <CardHeader className="bg-gradient-to-r from-[#002395] to-[#001875] text-white rounded-t-xl">
        <CardTitle className="text-xl">Calculateur de prêt immobilier</CardTitle>
      </CardHeader>
      <CardContent className="p-6 space-y-4">
        <div className="space-y-2">
          <div className="flex justify-between">
            <Label htmlFor="loan-amount">Montant du prêt</Label>
            <span className="font-medium">{formatCurrency(loanAmount)}</span>
          </div>
          <Slider
            id="loan-amount"
            min={50000}
            max={1000000}
            step={10000}
            value={[loanAmount]}
            onValueChange={(values) => handleLoanAmountChange(values[0])}
            className="my-4"
          />
          <div className="flex justify-between text-sm text-gray-500">
            <span>50 000 €</span>
            <span>1 000 000 €</span>
          </div>
        </div>

        <div className="space-y-2">
          <div className="flex justify-between">
            <Label htmlFor="interest-rate">Taux d'intérêt (%)</Label>
            <span className="font-medium">{interestRate.toFixed(2)}%</span>
          </div>
          <Slider
            id="interest-rate"
            min={0.5}
            max={10}
            step={0.1}
            value={[interestRate]}
            onValueChange={(values) => handleInterestRateChange(values[0])}
            className="my-4"
          />
          <div className="flex justify-between text-sm text-gray-500">
            <span>0.5%</span>
            <span>10%</span>
          </div>
        </div>

        <div className="space-y-2">
          <div className="flex justify-between">
            <Label htmlFor="loan-term">Durée (années)</Label>
            <span className="font-medium">{loanTerm} ans</span>
          </div>
          <Slider
            id="loan-term"
            min={5}
            max={30}
            step={1}
            value={[loanTerm]}
            onValueChange={(values) => handleLoanTermChange(values[0])}
            className="my-4"
          />
          <div className="flex justify-between text-sm text-gray-500">
            <span>5 ans</span>
            <span>30 ans</span>
          </div>
        </div>

        <div className="pt-4 border-t border-gray-200">
          <div className="text-center">
            <div className="text-sm text-gray-500 mb-1">Mensualité estimée</div>
            <div className="text-3xl font-bold text-[#002395]">{formatCurrency(monthlyPayment)}</div>
          </div>
        </div>

        <Link href="/contact">
          <Button className="w-full bg-[#002395] hover:bg-[#001875] text-white">Demander un devis personnalisé</Button>
        </Link>
      </CardContent>
    </Card>
  )
}
