"use client"

import { useState, useEffect, useRef } from "react"
import mapboxgl from "mapbox-gl"
import "mapbox-gl/dist/mapbox-gl.css"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"

// Données enrichies pour chaque arrondissement
const districtsData = [
  {
    id: "75001",
    name: "1er",
    center: [2.3417, 48.8625],
    vacancyRate: 1.8,
    averageOccupancy: 2.3,
    tenantProfile: "Cadres supérieurs, expatriés, touristes (location courte durée)",
    pricePerSqm: 13500,
    rentalYield: 3.0,
    description:
      "Quartier historique et touristique, idéal pour la location saisonnière et les investissements premium.",
  },
  {
    id: "75002",
    name: "2ème",
    center: [2.3444, 48.8685],
    vacancyRate: 2.1,
    averageOccupancy: 2.5,
    tenantProfile: "Jeunes actifs, cadres, touristes",
    pricePerSqm: 12800,
    rentalYield: 3.2,
    description: "Quartier central des affaires avec une forte demande locative des jeunes professionnels.",
  },
  {
    id: "75003",
    name: "3ème",
    center: [2.3621, 48.8631],
    vacancyRate: 1.9,
    averageOccupancy: 2.7,
    tenantProfile: "Cadres créatifs, couples sans enfants",
    pricePerSqm: 13200,
    rentalYield: 3.1,
    description: "Partie nord du Marais, très prisée pour son ambiance branchée et ses boutiques.",
  },
  {
    id: "75004",
    name: "4ème",
    center: [2.3553, 48.855],
    vacancyRate: 1.7,
    averageOccupancy: 2.4,
    tenantProfile: "Cadres supérieurs, couples, touristes",
    pricePerSqm: 13800,
    rentalYield: 2.9,
    description: "Cœur historique de Paris incluant l'Île Saint-Louis et une partie du Marais.",
  },
  {
    id: "75005",
    name: "5ème",
    center: [2.3457, 48.8448],
    vacancyRate: 2.0,
    averageOccupancy: 3.1,
    tenantProfile: "Étudiants, enseignants, familles",
    pricePerSqm: 12500,
    rentalYield: 3.3,
    description: "Quartier Latin, forte présence étudiante avec de nombreuses institutions académiques.",
  },
  {
    id: "75006",
    name: "6ème",
    center: [2.335, 48.8493],
    vacancyRate: 1.5,
    averageOccupancy: 2.8,
    tenantProfile: "Cadres supérieurs, professions libérales",
    pricePerSqm: 14200,
    rentalYield: 2.8,
    description: "Saint-Germain-des-Prés, quartier haut de gamme avec une forte valeur patrimoniale.",
  },
  {
    id: "75007",
    name: "7ème",
    center: [2.3175, 48.8574],
    vacancyRate: 1.6,
    averageOccupancy: 3.0,
    tenantProfile: "Familles aisées, diplomates, cadres dirigeants",
    pricePerSqm: 14500,
    rentalYield: 2.7,
    description: "Quartier prestigieux abritant la Tour Eiffel et de nombreuses ambassades.",
  },
  {
    id: "75008",
    name: "8ème",
    center: [2.3125, 48.8728],
    vacancyRate: 2.2,
    averageOccupancy: 2.6,
    tenantProfile: "Cadres dirigeants, expatriés, professions libérales",
    pricePerSqm: 13900,
    rentalYield: 2.9,
    description: "Quartier des Champs-Élysées et du Triangle d'Or, très prisé des entreprises.",
  },
  {
    id: "75009",
    name: "9ème",
    center: [2.3376, 48.8765],
    vacancyRate: 2.3,
    averageOccupancy: 2.5,
    tenantProfile: "Jeunes actifs, cadres créatifs, couples",
    pricePerSqm: 11200,
    rentalYield: 3.5,
    description: "Quartier en pleine transformation, notamment autour de Pigalle et rue des Martyrs.",
  },
  {
    id: "75010",
    name: "10ème",
    center: [2.3558, 48.8762],
    vacancyRate: 2.8,
    averageOccupancy: 2.4,
    tenantProfile: "Jeunes actifs, artistes, étudiants",
    pricePerSqm: 10300,
    rentalYield: 3.7,
    description: "Quartier cosmopolite en gentrification, notamment autour du Canal Saint-Martin.",
  },
  {
    id: "75011",
    name: "11ème",
    center: [2.3795, 48.8583],
    vacancyRate: 2.5,
    averageOccupancy: 2.3,
    tenantProfile: "Jeunes actifs, couples sans enfants",
    pricePerSqm: 10800,
    rentalYield: 3.8,
    description: "Quartier dynamique et branché, particulièrement autour de Bastille et Oberkampf.",
  },
  {
    id: "75012",
    name: "12ème",
    center: [2.3883, 48.8402],
    vacancyRate: 2.4,
    averageOccupancy: 3.2,
    tenantProfile: "Familles, cadres moyens et supérieurs",
    pricePerSqm: 10200,
    rentalYield: 3.6,
    description: "Quartier familial avec le Bois de Vincennes et le marché d'Aligre.",
  },
  {
    id: "75013",
    name: "13ème",
    center: [2.3561, 48.8322],
    vacancyRate: 2.7,
    averageOccupancy: 3.0,
    tenantProfile: "Étudiants, jeunes actifs, familles",
    pricePerSqm: 9800,
    rentalYield: 3.9,
    description: "Quartier en développement avec la BNF et Paris Rive Gauche.",
  },
  {
    id: "75014",
    name: "14ème",
    center: [2.3264, 48.8328],
    vacancyRate: 2.6,
    averageOccupancy: 3.1,
    tenantProfile: "Familles, cadres, étudiants",
    pricePerSqm: 10500,
    rentalYield: 3.5,
    description: "Quartier résidentiel calme avec une bonne qualité de vie.",
  },
  {
    id: "75015",
    name: "15ème",
    center: [2.3002, 48.8417],
    vacancyRate: 2.2,
    averageOccupancy: 3.3,
    tenantProfile: "Familles, cadres supérieurs",
    pricePerSqm: 11300,
    rentalYield: 3.3,
    description: "Plus grand arrondissement de Paris, très familial et résidentiel.",
  },
  {
    id: "75016",
    name: "16ème",
    center: [2.2769, 48.8637],
    vacancyRate: 2.0,
    averageOccupancy: 3.4,
    tenantProfile: "Cadres supérieurs, expatriés, familles aisées",
    pricePerSqm: 12500,
    rentalYield: 2.9,
    description: "Quartier huppé avec le Bois de Boulogne et de nombreuses écoles internationales.",
  },
  {
    id: "75017",
    name: "17ème",
    center: [2.3142, 48.8835],
    vacancyRate: 2.3,
    averageOccupancy: 3.0,
    tenantProfile: "Cadres, familles, professions libérales",
    pricePerSqm: 11800,
    rentalYield: 3.2,
    description: "Quartier contrasté entre le très chic Plaine Monceau et les Batignolles en plein essor.",
  },
  {
    id: "75018",
    name: "18ème",
    center: [2.3488, 48.8925],
    vacancyRate: 3.0,
    averageOccupancy: 2.2,
    tenantProfile: "Jeunes actifs, artistes, étudiants",
    pricePerSqm: 9600,
    rentalYield: 4.0,
    description: "Quartier contrasté entre Montmartre touristique et zones en gentrification.",
  },
  {
    id: "75019",
    name: "19ème",
    center: [2.382, 48.8819],
    vacancyRate: 3.2,
    averageOccupancy: 2.8,
    tenantProfile: "Familles, jeunes actifs, étudiants",
    pricePerSqm: 9200,
    rentalYield: 4.2,
    description: "Quartier en transformation avec le parc de la Villette et le bassin de la Villette.",
  },
  {
    id: "75020",
    name: "20ème",
    center: [2.3984, 48.865],
    vacancyRate: 3.1,
    averageOccupancy: 2.6,
    tenantProfile: "Étudiants, jeunes actifs, artistes",
    pricePerSqm: 9500,
    rentalYield: 4.1,
    description: "Quartier populaire en gentrification, notamment autour de Ménilmontant et Gambetta.",
  },
]

type DistrictData = {
  id: string
  name: string
  center: [number, number]
  vacancyRate: number
  averageOccupancy: number
  tenantProfile: string
  pricePerSqm: number
  rentalYield: number
  description: string
}

interface ParisDistrictsMapProps {
  mapboxToken: string
}

export default function ParisDistrictsMap({ mapboxToken }: ParisDistrictsMapProps) {
  const mapContainer = useRef<HTMLDivElement>(null)
  const map = useRef<mapboxgl.Map | null>(null)
  const [selectedDistrict, setSelectedDistrict] = useState<DistrictData | null>(null)
  const [activeTab, setActiveTab] = useState("vacancy")

  useEffect(() => {
    if (!mapContainer.current || map.current || !mapboxToken) return

    // Configuration du token Mapbox
    mapboxgl.accessToken = mapboxToken

    map.current = new mapboxgl.Map({
      container: mapContainer.current,
      style: "mapbox://styles/mapbox/light-v11",
      center: [2.3522, 48.8566], // Centre de Paris
      zoom: 11.5,
    })

    map.current.on("load", () => {
      if (!map.current) return

      // Ajouter les contrôles de navigation
      map.current.addControl(new mapboxgl.NavigationControl(), "top-right")

      // Charger les données GeoJSON des arrondissements de Paris
      // Note: Dans une application réelle, vous utiliseriez un fichier GeoJSON complet des arrondissements
      // Pour cette démonstration, nous utiliserons des marqueurs simples
      districtsData.forEach((district) => {
        const popup = new mapboxgl.Popup({ offset: 25 }).setHTML(`<strong>${district.name} arrondissement</strong>`)

        const el = document.createElement("div")
        el.className = "district-marker"
        el.style.width = "30px"
        el.style.height = "30px"
        el.style.borderRadius = "50%"
        el.style.backgroundColor = "rgba(0, 123, 255, 0.5)"
        el.style.border = "2px solid #007bff"
        el.style.display = "flex"
        el.style.justifyContent = "center"
        el.style.alignItems = "center"
        el.style.color = "white"
        el.style.fontWeight = "bold"
        el.style.cursor = "pointer"
        el.innerText = district.name

        el.addEventListener("click", () => {
          setSelectedDistrict(district)
        })

        new mapboxgl.Marker(el).setLngLat(district.center).setPopup(popup).addTo(map.current!)
      })
    })

    return () => {
      if (map.current) {
        map.current.remove()
        map.current = null
      }
    }
  }, [mapboxToken])

  // Fonction pour générer la couleur en fonction de la valeur
  const getColorForValue = (value: number, metric: string): string => {
    let min: number, max: number

    switch (metric) {
      case "vacancy":
        min = 1.5
        max = 3.2
        // Rouge pour taux élevé (mauvais), vert pour taux bas (bon)
        return `rgb(${Math.round((255 * (value - min)) / (max - min))}, ${Math.round(255 * (1 - (value - min) / (max - min)))}, 0)`
      case "occupancy":
        min = 2.2
        max = 3.4
        // Vert pour durée élevée (bon), rouge pour durée basse (mauvais)
        return `rgb(${Math.round(255 * (1 - (value - min) / (max - min)))}, ${Math.round((255 * (value - min)) / (max - min))}, 0)`
      case "yield":
        min = 2.7
        max = 4.2
        // Vert pour rendement élevé (bon), rouge pour rendement bas (mauvais)
        return `rgb(${Math.round(255 * (1 - (value - min) / (max - min)))}, ${Math.round((255 * (value - min)) / (max - min))}, 0)`
      default:
        return "#007bff"
    }
  }



  return (
    <div className="w-full">
      <Card className="mb-6">
        <CardHeader>
          <CardTitle>Carte Interactive des Arrondissements de Paris</CardTitle>
          <CardDescription>Explorez les données locatives et d'investissement par arrondissement</CardDescription>
        </CardHeader>
        <CardContent>
          <Tabs value={activeTab} onValueChange={setActiveTab} className="mb-4">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="vacancy">Taux de vacance</TabsTrigger>
              <TabsTrigger value="occupancy">Durée d'occupation</TabsTrigger>
              <TabsTrigger value="yield">Rendement locatif</TabsTrigger>
            </TabsList>
            <TabsContent value="vacancy" className="mt-2">
              <p className="text-sm text-muted-foreground mb-2">
                Le taux de vacance locative indique le pourcentage de logements vacants dans chaque arrondissement. Un
                taux bas (vert) indique une forte demande locative.
              </p>
              <div className="flex justify-between items-center text-xs mt-1 mb-4">
                <span>Faible vacance (1.5%)</span>
                <div className="w-2/3 h-2 mx-2 bg-gradient-to-r from-green-500 to-red-500 rounded"></div>
                <span>Forte vacance (3.2%)</span>
              </div>
            </TabsContent>
            <TabsContent value="occupancy" className="mt-2">
              <p className="text-sm text-muted-foreground mb-2">
                La durée moyenne d'occupation en années. Une durée plus longue (vert) indique une meilleure stabilité
                locative.
              </p>
              <div className="flex justify-between items-center text-xs mt-1 mb-4">
                <span>Courte durée (2.2 ans)</span>
                <div className="w-2/3 h-2 mx-2 bg-gradient-to-r from-red-500 to-green-500 rounded"></div>
                <span>Longue durée (3.4 ans)</span>
              </div>
            </TabsContent>
            <TabsContent value="yield" className="mt-2">
              <p className="text-sm text-muted-foreground mb-2">
                Le rendement locatif brut moyen. Un rendement élevé (vert) indique un meilleur retour sur
                investissement.
              </p>
              <div className="flex justify-between items-center text-xs mt-1 mb-4">
                <span>Faible rendement (2.7%)</span>
                <div className="w-2/3 h-2 mx-2 bg-gradient-to-r from-red-500 to-green-500 rounded"></div>
                <span>Fort rendement (4.2%)</span>
              </div>
            </TabsContent>
          </Tabs>

          <div ref={mapContainer} className="h-[500px] rounded-md border" />

          <div className="grid grid-cols-5 gap-2 mt-4">
            {districtsData.map((district) => {
              let value: number
              let label: string

              switch (activeTab) {
                case "vacancy":
                  value = district.vacancyRate
                  label = `${value}%`
                  break
                case "occupancy":
                  value = district.averageOccupancy
                  label = `${value} ans`
                  break
                case "yield":
                  value = district.rentalYield
                  label = `${value}%`
                  break
                default:
                  value = district.vacancyRate
                  label = `${value}%`
              }

              const color = getColorForValue(value, activeTab)

              return (
                <button
                  key={district.id}
                  className="p-2 rounded-md border text-center hover:bg-gray-100 transition-colors"
                  style={{ borderColor: color }}
                  onClick={() => {
                    setSelectedDistrict(district)
                    map.current?.flyTo({
                      center: district.center,
                      zoom: 13,
                      duration: 1000,
                    })
                  }}
                >
                  <div className="font-bold">{district.name}</div>
                  <div className="text-sm" style={{ color }}>
                    {label}
                  </div>
                </button>
              )
            })}
          </div>
        </CardContent>
      </Card>

      {selectedDistrict && (
        <Card>
          <CardHeader>
            <CardTitle>{selectedDistrict.name}ème arrondissement</CardTitle>
            <CardDescription>Données locales détaillées</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <h4 className="font-semibold mb-2">Données locatives</h4>
                <ul className="space-y-2">
                  <li className="flex justify-between">
                    <span className="text-muted-foreground">Taux de vacance:</span>
                    <span className="font-medium">{selectedDistrict.vacancyRate}%</span>
                  </li>
                  <li className="flex justify-between">
                    <span className="text-muted-foreground">Durée moyenne d'occupation:</span>
                    <span className="font-medium">{selectedDistrict.averageOccupancy} ans</span>
                  </li>
                  <li className="flex justify-between">
                    <span className="text-muted-foreground">Profil type des locataires:</span>
                    <span className="font-medium">{selectedDistrict.tenantProfile}</span>
                  </li>
                </ul>
              </div>
              <div>
                <h4 className="font-semibold mb-2">Données d'investissement</h4>
                <ul className="space-y-2">
                  <li className="flex justify-between">
                    <span className="text-muted-foreground">Prix moyen au m²:</span>
                    <span className="font-medium">{selectedDistrict.pricePerSqm.toLocaleString()} €</span>
                  </li>
                  <li className="flex justify-between">
                    <span className="text-muted-foreground">Rendement locatif brut:</span>
                    <span className="font-medium">{selectedDistrict.rentalYield}%</span>
                  </li>
                </ul>
              </div>
            </div>
            <div className="mt-4">
              <h4 className="font-semibold mb-2">Description</h4>
              <p className="text-muted-foreground">{selectedDistrict.description}</p>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  )
}
