"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Slider } from "@/components/ui/slider"
import { Label } from "@/components/ui/label"
import { Info } from "lucide-react"
import Link from "next/link"

export default function RentabiliteCalculator() {
  // Prix d'achat et financement
  const [prixAchat, setPrixAchat] = useState(300000)
  const [fraisNotaire, setFraisNotaire] = useState(21000) // 7% par défaut
  const [fraisAgence, setFraisAgence] = useState(15000) // 5% par défaut
  const [travaux, setTravaux] = useState(0)
  const [apport, setApport] = useState(60000) // 20% par défaut

  // Revenus locatifs
  const [loyerMensuel, setLoyerMensuel] = useState(1200)
  const [chargesNonRecuperables, setChargesNonRecuperables] = useState(600) // 50€/mois par défaut
  const [tauxVacance, setTauxVacance] = useState(5) // 5% par défaut
  const [tauxImpaye, setTauxImpaye] = useState(1) // 1% par défaut

  // Financement
  const [tauxInteret, setTauxInteret] = useState(3.5)
  const [dureeCredit, setDureeCredit] = useState(20)

  // Résultats calculés
  const [investissementTotal, setInvestissementTotal] = useState(0)
  const [mensualiteCredit, setMensualiteCredit] = useState(0)
  const [cashflowMensuel, setCashflowMensuel] = useState(0)
  const [rentabiliteBrute, setRentabiliteBrute] = useState(0)
  const [rentabiliteNette, setRentabiliteNette] = useState(0)

  // Calculer tous les résultats lorsque les inputs changent
  useEffect(() => {
    // Calcul de l'investissement total
    const investissementTotal = prixAchat + fraisNotaire + fraisAgence + travaux
    setInvestissementTotal(investissementTotal)

    // Calcul du montant emprunté
    const montantEmprunte = investissementTotal - apport

    // Calcul de la mensualité du crédit
    const tauxMensuel = tauxInteret / 100 / 12
    const nombreMensualites = dureeCredit * 12
    const mensualite =
      montantEmprunte > 0
        ? (montantEmprunte * tauxMensuel * Math.pow(1 + tauxMensuel, nombreMensualites)) /
          (Math.pow(1 + tauxMensuel, nombreMensualites) - 1)
        : 0
    setMensualiteCredit(mensualite)

    // Calcul des revenus locatifs annuels avec prise en compte de la vacance et des impayés
    const revenuLocatifAnnuel = loyerMensuel * 12 * (1 - tauxVacance / 100) * (1 - tauxImpaye / 100)

    // Calcul des charges annuelles
    const chargesAnnuelles = chargesNonRecuperables * 12

    // Calcul du cash-flow mensuel
    const cashflowAnnuel = revenuLocatifAnnuel - chargesAnnuelles - mensualite * 12
    setCashflowMensuel(cashflowAnnuel / 12)

    // Calcul de la rentabilité brute (avant charges et crédit)
    const rentabiliteBrute = ((loyerMensuel * 12) / investissementTotal) * 100
    setRentabiliteBrute(rentabiliteBrute)

    // Calcul de la rentabilité nette (après charges, avant crédit)
    const rentabiliteNette = ((revenuLocatifAnnuel - chargesAnnuelles) / investissementTotal) * 100
    setRentabiliteNette(rentabiliteNette)
  }, [
    prixAchat,
    fraisNotaire,
    fraisAgence,
    travaux,
    apport,
    loyerMensuel,
    chargesNonRecuperables,
    tauxVacance,
    tauxImpaye,
    tauxInteret,
    dureeCredit,
  ])

  // Formater les nombres en euros
  const formatEuro = (value) => {
    return new Intl.NumberFormat("fr-FR", {
      style: "currency",
      currency: "EUR",
      maximumFractionDigits: 0,
    }).format(value)
  }

  // Formater les pourcentages
  const formatPercent = (value) => {
    return new Intl.NumberFormat("fr-FR", {
      style: "percent",
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(value / 100)
  }

  return (
    <div className="space-y-8">
      <div className="grid md:grid-cols-2 gap-8">
        {/* Colonne de gauche - Paramètres d'achat */}
        <div className="space-y-6">
          <h3 className="text-xl font-bold text-[#002395]">Paramètres d'achat</h3>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="prix-achat">Prix d'achat</Label>
              <span className="font-medium">{formatEuro(prixAchat)}</span>
            </div>
            <Slider
              id="prix-achat"
              min={50000}
              max={1000000}
              step={5000}
              value={[prixAchat]}
              onValueChange={(values) => setPrixAchat(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="frais-notaire">Frais de notaire</Label>
              <span className="font-medium">{formatEuro(fraisNotaire)}</span>
            </div>
            <Slider
              id="frais-notaire"
              min={0}
              max={Math.round(prixAchat * 0.1)}
              step={1000}
              value={[fraisNotaire]}
              onValueChange={(values) => setFraisNotaire(values[0])}
              className="my-2"
            />
            <div className="text-xs text-gray-500 text-right">
              Environ {Math.round((fraisNotaire / prixAchat) * 100)}% du prix d'achat
            </div>
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="frais-agence">Frais d'agence</Label>
              <span className="font-medium">{formatEuro(fraisAgence)}</span>
            </div>
            <Slider
              id="frais-agence"
              min={0}
              max={Math.round(prixAchat * 0.1)}
              step={1000}
              value={[fraisAgence]}
              onValueChange={(values) => setFraisAgence(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="travaux">Budget travaux</Label>
              <span className="font-medium">{formatEuro(travaux)}</span>
            </div>
            <Slider
              id="travaux"
              min={0}
              max={100000}
              step={1000}
              value={[travaux]}
              onValueChange={(values) => setTravaux(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="apport">Apport personnel</Label>
              <span className="font-medium">{formatEuro(apport)}</span>
            </div>
            <Slider
              id="apport"
              min={0}
              max={investissementTotal}
              step={5000}
              value={[apport]}
              onValueChange={(values) => setApport(values[0])}
              className="my-2"
            />
            <div className="text-xs text-gray-500 text-right">
              {Math.round((apport / investissementTotal) * 100)}% de l'investissement total
            </div>
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="taux-interet">Taux d'intérêt (%)</Label>
              <span className="font-medium">{tauxInteret.toFixed(2)}%</span>
            </div>
            <Slider
              id="taux-interet"
              min={1}
              max={7}
              step={0.1}
              value={[tauxInteret]}
              onValueChange={(values) => setTauxInteret(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="duree-credit">Durée du crédit (années)</Label>
              <span className="font-medium">{dureeCredit} ans</span>
            </div>
            <Slider
              id="duree-credit"
              min={5}
              max={30}
              step={1}
              value={[dureeCredit]}
              onValueChange={(values) => setDureeCredit(values[0])}
              className="my-2"
            />
          </div>
        </div>

        {/* Colonne de droite - Paramètres locatifs et résultats */}
        <div className="space-y-6">
          <h3 className="text-xl font-bold text-[#002395]">Paramètres locatifs</h3>

          <div className="space-y-2">
            <div className="flex justify-between">
              <Label htmlFor="loyer-mensuel">Loyer mensuel</Label>
              <span className="font-medium">{formatEuro(loyerMensuel)}</span>
            </div>
            <Slider
              id="loyer-mensuel"
              min={300}
              max={5000}
              step={50}
              value={[loyerMensuel]}
              onValueChange={(values) => setLoyerMensuel(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <div className="flex items-center gap-1">
                <Label htmlFor="charges-non-recuperables">Charges annuelles non récupérables</Label>
                <div
                  className="text-gray-400 cursor-help"
                  title="Taxe foncière, charges de copropriété non récupérables, assurances, etc."
                >
                  <Info className="h-4 w-4" />
                </div>
              </div>
              <span className="font-medium">{formatEuro(chargesNonRecuperables * 12)}</span>
            </div>
            <Slider
              id="charges-non-recuperables"
              min={0}
              max={5000}
              step={100}
              value={[chargesNonRecuperables * 12]}
              onValueChange={(values) => setChargesNonRecuperables(values[0] / 12)}
              className="my-2"
            />
            <div className="text-xs text-gray-500 text-right">Soit {formatEuro(chargesNonRecuperables)} par mois</div>
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <div className="flex items-center gap-1">
                <Label htmlFor="taux-vacance">Taux de vacance locative</Label>
                <div
                  className="text-gray-400 cursor-help"
                  title="Pourcentage du temps où le bien est inoccupé entre deux locations"
                >
                  <Info className="h-4 w-4" />
                </div>
              </div>
              <span className="font-medium">{tauxVacance}%</span>
            </div>
            <Slider
              id="taux-vacance"
              min={0}
              max={20}
              step={1}
              value={[tauxVacance]}
              onValueChange={(values) => setTauxVacance(values[0])}
              className="my-2"
            />
          </div>

          <div className="space-y-2">
            <div className="flex justify-between">
              <div className="flex items-center gap-1">
                <Label htmlFor="taux-impaye">Taux d'impayés</Label>
                <div className="text-gray-400 cursor-help" title="Estimation du risque de loyers impayés">
                  <Info className="h-4 w-4" />
                </div>
              </div>
              <span className="font-medium">{tauxImpaye}%</span>
            </div>
            <Slider
              id="taux-impaye"
              min={0}
              max={10}
              step={0.5}
              value={[tauxImpaye]}
              onValueChange={(values) => setTauxImpaye(values[0])}
              className="my-2"
            />
          </div>

          <div className="mt-8 pt-6 border-t border-gray-200">
            <h3 className="text-xl font-bold text-[#002395] mb-4">Résultats</h3>

            <div className="space-y-4 bg-gray-50 p-4 rounded-lg">
              <div className="flex justify-between">
                <span>Investissement total</span>
                <span className="font-bold">{formatEuro(investissementTotal)}</span>
              </div>
              <div className="flex justify-between">
                <span>Montant emprunté</span>
                <span className="font-bold">{formatEuro(investissementTotal - apport)}</span>
              </div>
              <div className="flex justify-between">
                <span>Mensualité du crédit</span>
                <span className="font-bold">{formatEuro(mensualiteCredit)}</span>
              </div>
              <div className="flex justify-between">
                <span>Cash-flow mensuel</span>
                <span className={`font-bold ${cashflowMensuel >= 0 ? "text-green-600" : "text-red-600"}`}>
                  {formatEuro(cashflowMensuel)}
                </span>
              </div>
              <div className="flex justify-between pt-4 border-t border-gray-200">
                <span>Rentabilité brute</span>
                <span className="font-bold text-[#002395]">{formatPercent(rentabiliteBrute)}</span>
              </div>
              <div className="flex justify-between">
                <span>Rentabilité nette</span>
                <span className="font-bold text-[#002395]">{formatPercent(rentabiliteNette)}</span>
              </div>
            </div>
          </div>
        </div>
      </div>

      <Link href="/contact">
        <Button className="w-full bg-[#002395] hover:bg-[#001875] text-white">
          Obtenir une analyse personnalisée de votre projet
        </Button>
      </Link>
    </div>
  )
}
