"use client"

import { useEffect, useState } from "react"

interface Heading {
  id: string
  text: string
  level: number
}

interface TableOfContentsProps {
  headings: Heading[]
}

export default function SidebarTableOfContents({ headings }: TableOfContentsProps) {
  const [activeId, setActiveId] = useState<string>("")
  const [isOpen, setIsOpen] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            setActiveId(entry.target.id)
          }
        })
      },
      { rootMargin: "-100px 0px -80% 0px" },
    )

    // Observe all section headings
    headings.forEach((heading) => {
      const element = document.getElementById(heading.id)
      if (element) {
        observer.observe(element)
      }
    })

    return () => {
      headings.forEach((heading) => {
        const element = document.getElementById(heading.id)
        if (element) {
          observer.unobserve(element)
        }
      })
    }
  }, [headings])

  return (
    <div className="relative">
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center gap-2 text-sm font-medium text-[#002395] hover:text-[#001875] transition-colors"
      >
        <span>Table des matières</span>
        <span className={`transition-transform ${isOpen ? "rotate-180" : ""}`}>▼</span>
      </button>

      {isOpen && (
        <nav className="absolute z-10 mt-2 bg-white rounded-lg shadow-lg border border-gray-200 p-4 w-64">
          <ul className="space-y-2">
            {headings.map((heading) => (
              <li key={heading.id} className={`${heading.level === 3 ? "ml-4" : ""}`}>
                <a
                  href={`#${heading.id}`}
                  className={`block py-1 border-l-2 pl-3 text-sm hover:text-[#002395] transition-colors ${
                    activeId === heading.id
                      ? "border-[#002395] text-[#002395] font-medium"
                      : "border-gray-200 text-gray-600"
                  }`}
                  onClick={(e) => {
                    e.preventDefault()
                    document.getElementById(heading.id)?.scrollIntoView({
                      behavior: "smooth",
                    })
                    // Update URL without page reload
                    window.history.pushState(null, "", `#${heading.id}`)
                    setIsOpen(false)
                  }}
                >
                  {heading.text}
                </a>
              </li>
            ))}
          </ul>
        </nav>
      )}
    </div>
  )
}
