"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Label } from "@/components/ui/label"
import { Check, Copy } from "lucide-react"

export default function TemplateGenerator() {
  const [departement, setDepartement] = useState("")
  const [code, setCode] = useState("")
  const [slug, setSlug] = useState("")
  const [description, setDescription] = useState("")
  const [copied, setCopied] = useState(false)

  const generateTemplate = () => {
    const template = `import type { Metadata } from "next"
import Image from "next/image"
import Link from "next/link"
import { Check } from 'lucide-react'
import PageHeader from "@/components/page-header"
import CTASection from "@/components/cta-section"
import SmartContactForm from "@/components/smart-contact-form"

export const metadata: Metadata = {
  title: "Administrateur de biens ${departement ? `en ${departement}` : ""} ${code ? `(${code})` : ""} | DOGES",
  description:
    "Services d'administration de biens immobiliers ${departement ? `en ${departement}` : ""}. Gestion locative, syndic de copropriété et conseil patrimonial par des experts locaux.",
  keywords:
    "administrateur de biens ${departement}, gestion immobilière ${code}, syndic copropriété ${departement}, gestionnaire patrimonial ${departement}, DOGES",
}

export default function AdministrateurBiens${slug.charAt(0).toUpperCase() + slug.slice(1)}Page() {
  return (
    <>
      <PageHeader
        title="Administrateur de biens ${departement ? `en ${departement}` : ""} ${code ? `(${code})` : ""}"
        description="Une gestion professionnelle pour votre patrimoine immobilier ${departement ? `en ${departement}` : ""}"
        breadcrumbs={[
          { label: "Services", href: "/services" },
          { label: "Administrateur de biens en Île-de-France", href: "/services/administrateur-biens-ile-de-france" },
          { label: "${departement}", href: "/services/administrateur-biens-${slug}" },
        ]}
        backgroundImage="/parisian-street.png"
      />

      {/* Présentation du service */}
      <section className="py-16">
        <div className="container mx-auto px-4">
          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl font-bold mb-6 text-[#002395]">
                Une expertise complète en administration de biens ${departement ? `en ${departement}` : ""}
              </h2>
              <p className="text-lg text-gray-700 mb-6">
                DOGES propose des services d'administration de biens sur l'ensemble ${departement ? `de la ${departement}` : ""}, adaptés aux
                spécificités de chaque commune. Nos équipes d'experts locaux assurent une gestion de
                proximité efficace et personnalisée.
              </p>
              <p className="text-lg text-gray-700 mb-6">
                ${
                  description ||
                  `Qu'il s'agisse de gestion locative, de syndic de copropriété ou de conseil patrimonial, nous intervenons
                dans toutes les communes ${departement ? `de ${departement}` : ""} avec la même exigence de qualité et de professionnalisme.`
                }
              </p>
              <div className="space-y-4">
                {[
                  "Services complets d'administration de biens sur l'ensemble du territoire",
                  "Équipes spécialisées par secteur pour une expertise locale précise",
                  "Gestion adaptée aux spécificités du marché immobilier local",
                  "Solutions sur mesure pour tous types de patrimoine (résidentiel, commercial, mixte)",
                  "Veille réglementaire locale pour une conformité totale",
                  "Suivi personnalisé avec un interlocuteur dédié",
                ].map((point, index) => (
                  <div key={index} className="flex items-start">
                    <div className="mt-1 mr-3 bg-[#002395] rounded-full p-1">
                      <Check className="h-4 w-4 text-white" />
                    </div>
                    <p>{point}</p>
                  </div>
                ))}
              </div>
            </div>
            <div className="relative h-[500px] rounded-2xl overflow-hidden">
              <Image
                src="/parisian-street.png"
                alt="Administrateur de biens ${departement ? `en ${departement}` : ""} DOGES"
                fill
                className="object-cover"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Reste du contenu... */}
      
      <CTASection
        title="Confiez l'administration de vos biens à nos experts ${departement ? `en ${departement}` : ""}"
        description="Bénéficiez d'un accompagnement personnalisé pour la gestion de votre patrimoine immobilier ${departement ? `en ${departement}` : ""}."
        buttonText="Demander un devis gratuit"
        buttonLink="/contact"
      />

      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="max-w-3xl mx-auto">
            <h2 className="text-3xl font-bold mb-8 text-center text-[#002395]">
              Contactez notre équipe d'administrateurs de biens ${departement ? `en ${departement}` : ""}
            </h2>
            <p className="text-lg text-gray-700 mb-8 text-center">
              Nos experts sont à votre disposition pour répondre à vos questions et vous proposer une solution
              d'administration adaptée à votre patrimoine ${departement ? `en ${departement}` : ""}.
            </p>
            <SmartContactForm />
          </div>
        </div>
      </section>
    </>
  )
}`

    return template
  }

  const handleCopy = () => {
    const template = generateTemplate()
    navigator.clipboard.writeText(template)
    setCopied(true)
    setTimeout(() => setCopied(false), 2000)
  }

  return (
    <div className="max-w-4xl mx-auto p-6 bg-white rounded-lg shadow-md">
      <h2 className="text-2xl font-bold mb-6 text-[#002395]">
        Générateur de template pour pages d'administrateur de biens
      </h2>

      <div className="grid gap-6 mb-6">
        <div>
          <Label htmlFor="departement">Nom du département ou de la ville</Label>
          <Input
            id="departement"
            value={departement}
            onChange={(e) => setDepartement(e.target.value)}
            placeholder="Ex: Yvelines, Val-de-Marne, Versailles..."
          />
        </div>

        <div>
          <Label htmlFor="code">Code département (optionnel)</Label>
          <Input id="code" value={code} onChange={(e) => setCode(e.target.value)} placeholder="Ex: 78, 94..." />
        </div>

        <div>
          <Label htmlFor="slug">Slug pour l'URL</Label>
          <Input
            id="slug"
            value={slug}
            onChange={(e) => setSlug(e.target.value)}
            placeholder="Ex: yvelines, val-de-marne, versailles..."
          />
        </div>

        <div>
          <Label htmlFor="description">Description personnalisée (optionnel)</Label>
          <Textarea
            id="description"
            value={description}
            onChange={(e) => setDescription(e.target.value)}
            placeholder="Laissez vide pour utiliser la description par défaut"
            rows={3}
          />
        </div>
      </div>

      <div className="flex justify-end">
        <Button onClick={handleCopy} className="flex items-center gap-2">
          {copied ? (
            <>
              <Check className="h-4 w-4" /> Copié !
            </>
          ) : (
            <>
              <Copy className="h-4 w-4" /> Copier le template
            </>
          )}
        </Button>
      </div>
    </div>
  )
}
