"use client"

import Image from "next/image"
import { useState } from "react"

type VilleImageProps = {
  ville: string
  slug: string
  className?: string
  height?: number
  width?: number
  priority?: boolean
}

export default function VilleImage({
  ville,
  slug,
  className = "",
  height = 400,
  width = 600,
  priority = false,
}: VilleImageProps) {
  const [imageError, setImageError] = useState(false)

  // Utiliser des images locales stockées dans le dossier public
  const getImageForVille = (slug: string) => {
    try {
      // Essayer d'utiliser une image locale spécifique à la ville
      return `/villes/${slug}.jpg`
    } catch (error) {
      // Fallback vers une image par défaut
      return `/villes/default-ville.png`
    }
  }

  return (
    <div className={`relative overflow-hidden rounded-lg ${className}`}>
      <Image
        src={getImageForVille(slug) || "/placeholder.svg"}
        alt={`Vue de ${ville}`}
        width={width}
        height={height}
        className="w-full h-full object-cover"
        onError={() => {
          setImageError(true)
          // Fallback en cas d'erreur
          const imgElements = document.querySelectorAll(`img[alt="Vue de ${ville}"]`)
          imgElements.forEach((img: HTMLImageElement) => {
            img.src = "/villes/default-ville.png"
          })
        }}
        priority={priority}
      />
      {imageError && (
        <div className="absolute inset-0 flex items-center justify-center bg-black/30">
          <h3 className="text-white text-xl font-bold px-4 text-center">{ville}</h3>
        </div>
      )}
    </div>
  )
}
