"use client"

import { useEffect, useState, useRef } from "react"

interface ImagePreloadResult {
  loaded: boolean
  error: boolean
}

export function useImagePreloader(imageSources: string[]): Record<string, ImagePreloadResult> {
  const [loadedImages, setLoadedImages] = useState<Record<string, ImagePreloadResult>>({})

  // Utiliser useRef pour stocker les sources d'images et éviter les re-rendus
  const imageSourcesRef = useRef<string[]>([])

  // Utiliser useRef pour suivre les instances d'Image
  const imageInstancesRef = useRef<Record<string, HTMLImageElement>>({})

  useEffect(() => {
    // Vérifier si les sources ont changé pour éviter les re-rendus inutiles
    const sourcesChanged =
      imageSources.length !== imageSourcesRef.current.length ||
      imageSources.some((src, i) => imageSourcesRef.current[i] !== src)

    if (!sourcesChanged) return

    // Mettre à jour la référence des sources
    imageSourcesRef.current = [...imageSources]

    // Nettoyer les anciennes instances d'images
    Object.values(imageInstancesRef.current).forEach((img) => {
      img.onload = null
      img.onerror = null
    })
    imageInstancesRef.current = {}

    // Initialiser l'état pour toutes les images
    const initialState = imageSources.reduce(
      (acc, src) => {
        acc[src] = { loaded: false, error: false }
        return acc
      },
      {} as Record<string, ImagePreloadResult>,
    )

    setLoadedImages(initialState)

    // Précharger toutes les images
    imageSources.forEach((src) => {
      // Éviter de recréer une instance si elle existe déjà
      if (!imageInstancesRef.current[src]) {
        const img = new Image()
        imageInstancesRef.current[src] = img

        img.onload = () => {
          // Utiliser un callback fonctionnel pour éviter les problèmes de fermeture
          setLoadedImages((prev) => ({
            ...prev,
            [src]: { loaded: true, error: false },
          }))
        }

        img.onerror = () => {
          setLoadedImages((prev) => ({
            ...prev,
            [src]: { loaded: false, error: true },
          }))
        }

        img.crossOrigin = "anonymous"
        img.src = src
      }
    })

    // Nettoyage lors du démontage du composant
    return () => {
      Object.values(imageInstancesRef.current).forEach((img) => {
        img.onload = null
        img.onerror = null
      })
    }
  }, [imageSources]) // Dépendance uniquement sur imageSources

  return loadedImages
}

// Hook pour précharger les images de services
export function useServiceImagesPreloader() {
  // Utiliser useRef pour éviter de recréer le tableau à chaque rendu
  const serviceImagesRef = useRef([
    "/modern-apartment-interior.png",
    "/real-estate-handshake.png",
    "/rental-income-protection-security.png",
    "/landlord-insurance-property-protection.png",
    "/real-estate-accounting-documents.png",
    "/legal-real-estate-professional-contract.png",
    "/service-commercial-immobilier.png",
    "/professionnels-immobilier-reunion.png",
  ])

  return useImagePreloader(serviceImagesRef.current)
}
