import type { UserProfile } from "./chatbot-profile-detector"

export interface ConversationStep {
  id: string
  message: string
  options?: string[]
  qualification?: string
  action?: string
  nextSteps?: Record<string, string>
}

export interface ConversationFlow {
  [stepId: string]: ConversationStep
}

export const conversationFlows: Record<UserProfile, ConversationFlow> = {
  acheteur_primo: {
    // Flux existant...
  },

  vendeur: {
    // Flux existant...
  },

  locataire: {
    // Flux existant...
  },

  proprietaire_bailleur: {
    // Flux existant...
  },

  investisseur: {
    // Flux existant...
  },

  professionnel: {
    // Flux existant...
  },

  service_inquiry: {
    welcome: {
      id: "welcome",
      message: "Je vais vous renseigner sur nos services. Quel service vous intéresse ?",
      options: [
        "Administrateur de Biens",
        "Garantie Loyers Impayés",
        "Assurance PNO",
        "Comptabilité Mandants",
        "Service Juridique",
        "Syndic de Copropriété",
        "Tous nos services",
      ],
      nextSteps: {
        "Administrateur de Biens": "admin_biens_details",
        "Garantie Loyers Impayés": "gli_details",
        "Assurance PNO": "pno_details",
        "Comptabilité Mandants": "comptabilite_details",
        "Service Juridique": "juridique_details",
        "Syndic de Copropriété": "syndic_details",
        "Tous nos services": "all_services",
      },
    },
    admin_biens_details: {
      id: "admin_biens_details",
      message:
        "Notre service d'Administration de Biens assure la gestion complète de votre patrimoine immobilier : gestion locative, comptabilité, suivi technique, valorisation patrimoniale. Dans quelle zone géographique se situe votre bien ?",
      options: ["Paris", "Hauts-de-Seine", "Île-de-France", "Immeubles haussmanniens", "Autre secteur"],
      nextSteps: {
        Paris: "admin_biens_paris",
        "Hauts-de-Seine": "admin_biens_92",
        "Île-de-France": "admin_biens_idf",
        "Immeubles haussmanniens": "admin_biens_haussmann",
        "Autre secteur": "admin_biens_autre",
      },
    },
    admin_biens_paris: {
      id: "admin_biens_paris",
      message:
        "Notre expertise en Administration de Biens à Paris comprend : gestion locative premium, suivi technique personnalisé, valorisation patrimoniale, comptabilité dédiée. Dans quel arrondissement se situe votre bien ?",
      options: [
        "Paris Centre (1-4)",
        "Paris 5-6-7",
        "Paris 8-16-17",
        "Paris 9-10-18",
        "Paris 11-12-20",
        "Paris 13-14-15",
        "Paris 19",
        "Tous arrondissements",
      ],
      action: "show_admin_biens_paris",
      nextSteps: {},
    },
    admin_biens_92: {
      id: "admin_biens_92",
      message:
        "Notre service d'Administration de Biens dans les Hauts-de-Seine (92) offre une gestion de proximité avec des équipes dédiées à ce secteur dynamique. Souhaitez-vous plus d'informations sur nos prestations dans cette zone ?",
      options: ["Tarifs et conditions", "Nos équipes locales", "Prendre rendez-vous", "Autres services"],
      action: "show_admin_biens_92",
      nextSteps: {},
    },
    admin_biens_idf: {
      id: "admin_biens_idf",
      message:
        "Notre service d'Administration de Biens en Île-de-France couvre l'ensemble de la région avec des équipes spécialisées par secteur. Nous assurons une gestion de proximité et une connaissance approfondie des marchés locaux.",
      options: ["Tarifs et conditions", "Nos équipes locales", "Prendre rendez-vous", "Autres services"],
      action: "show_admin_biens_idf",
      nextSteps: {},
    },
    admin_biens_haussmann: {
      id: "admin_biens_haussmann",
      message:
        "Notre expertise spécifique pour les immeubles haussmanniens comprend : connaissance des spécificités architecturales, gestion des parties communes classées, valorisation patrimoniale, et respect des contraintes historiques.",
      options: ["Tarifs spécifiques", "Expertise technique", "Prendre rendez-vous", "Autres services"],
      action: "show_admin_biens_haussmann",
      nextSteps: {},
    },
    admin_biens_autre: {
      id: "admin_biens_autre",
      message:
        "Nous proposons nos services d'Administration de Biens sur mesure pour votre secteur. Notre équipe s'adapte à vos besoins spécifiques et aux particularités locales de votre patrimoine.",
      options: ["Tarifs personnalisés", "Prendre rendez-vous", "Autres services"],
      action: "show_admin_biens_autre",
      nextSteps: {},
    },
    gli_details: {
      id: "gli_details",
      message:
        "Notre Garantie Loyers Impayés vous protège contre les risques locatifs : prise en charge des loyers impayés jusqu'à 100 000€, frais de contentieux, dégradations. Souhaitez-vous plus d'informations ?",
      options: ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
      action: "show_gli_info",
      nextSteps: {},
    },
    pno_details: {
      id: "pno_details",
      message:
        "L'Assurance Propriétaire Non Occupant protège votre bien : dégâts des eaux, incendie, vol, responsabilité civile. Couverture adaptée aux biens locatifs. Souhaitez-vous plus d'informations ?",
      options: ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
      action: "show_pno_info",
      nextSteps: {},
    },
    comptabilite_details: {
      id: "comptabilite_details",
      message:
        "Notre service Comptabilité Mandants assure la tenue complète de vos comptes immobiliers : suivi recettes/dépenses, décomptes annuels, déclarations fiscales, optimisation fiscale. Souhaitez-vous plus d'informations ?",
      options: ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
      action: "show_comptabilite_info",
      nextSteps: {},
    },
    juridique_details: {
      id: "juridique_details",
      message:
        "Notre service juridique vous accompagne : contentieux locatifs, procédures d'expulsion, recouvrement, conseils juridiques spécialisés. Souhaitez-vous plus d'informations ?",
      options: ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
      action: "show_juridique_info",
      nextSteps: {},
    },
    syndic_details: {
      id: "syndic_details",
      message:
        "Notre service de syndic assure la gestion complète de votre copropriété : assemblées générales, suivi travaux, gestion charges, relation copropriétaires. Souhaitez-vous plus d'informations ?",
      options: ["Tarifs et conditions", "Comment ça marche", "Prendre rendez-vous", "Autres services"],
      action: "show_syndic_info",
      nextSteps: {},
    },
    all_services: {
      id: "all_services",
      message:
        "Voici tous nos services : Administration de Biens, Gestion Locative, Transaction, Comptabilité Mandants, Garantie Loyers Impayés, Assurance PNO, Service Juridique, Syndic de Copropriété. Lequel vous intéresse ?",
      options: [
        "Administrateur de Biens",
        "Garantie Loyers Impayés",
        "Assurance PNO",
        "Comptabilité Mandants",
        "Service Juridique",
        "Prendre rendez-vous",
      ],
      nextSteps: {
        "Administrateur de Biens": "admin_biens_details",
        "Garantie Loyers Impayés": "gli_details",
        "Assurance PNO": "pno_details",
        "Comptabilité Mandants": "comptabilite_details",
        "Service Juridique": "juridique_details",
      },
    },
  },

  unknown: {
    welcome: {
      id: "welcome",
      message: "Je vais vous orienter vers le bon service. Pouvez-vous me préciser votre projet ?",
      options: [
        "Acheter",
        "Vendre",
        "Chercher un logement",
        "Louer mon bien",
        "Investir",
        "Professionnel",
        "Nos services",
        "Autre",
      ],
      nextSteps: {
        Acheter: "redirect_acheteur",
        Vendre: "redirect_vendeur",
        "Chercher un logement": "redirect_locataire",
        "Louer mon bien": "redirect_proprietaire_bailleur",
        Investir: "redirect_investisseur",
        Professionnel: "redirect_professionnel",
        "Nos services": "redirect_services",
        Autre: "contact",
      },
    },
    redirect_services: {
      id: "redirect_services",
      message: "Voici nos principaux services. Lequel vous intéresse ?",
      options: [
        "Administrateur de Biens",
        "Garantie Loyers Impayés",
        "Assurance PNO",
        "Comptabilité Mandants",
        "Service Juridique",
        "Syndic de Copropriété",
        "Gestion Locative",
        "Transaction Immobilière",
      ],
      nextSteps: {},
    },
  },
}

export function getWelcomeMessage(profile: UserProfile): ConversationStep {
  return conversationFlows[profile]?.welcome || conversationFlows.unknown.welcome
}

export function getNextStep(
  profile: UserProfile,
  currentStepId: string,
  selectedOption: string,
): ConversationStep | null {
  const flow = conversationFlows[profile]
  const currentStep = flow?.[currentStepId]

  if (!currentStep?.nextSteps) return null

  const nextStepId = currentStep.nextSteps[selectedOption]
  if (!nextStepId) return null

  return flow[nextStepId] || null
}

export function getStepById(profile: UserProfile, stepId: string): ConversationStep | null {
  return conversationFlows[profile]?.[stepId] || null
}
