const fs = require("fs")
const path = require("path")

function findFilesWithIncorrectImports(dir, fileList = []) {
  const files = fs.readdirSync(dir)

  files.forEach((file) => {
    const filePath = path.join(dir, file)
    const stat = fs.statSync(filePath)

    if (stat.isDirectory()) {
      // Skip node_modules and .next directories
      if (file !== "node_modules" && file !== ".next" && file !== ".git") {
        findFilesWithIncorrectImports(filePath, fileList)
      }
    } else if (file.endsWith(".tsx") || file.endsWith(".ts") || file.endsWith(".jsx") || file.endsWith(".js")) {
      const content = fs.readFileSync(filePath, "utf8")
      if (content.includes("next/Image")) {
        fileList.push(filePath)
      }
    }
  })

  return fileList
}

function fixImageImports(filePath) {
  let content = fs.readFileSync(filePath, "utf8")
  const originalContent = content

  // Replace all instances of 'next/Image' with 'next/image'
  content = content.replace(/from\s+['"]next\/Image['"]/g, 'from "next/image"')
  content = content.replace(/import\s+['"]next\/Image['"]/g, 'import "next/image"')

  if (content !== originalContent) {
    fs.writeFileSync(filePath, content, "utf8")
    console.log(`✅ Fixed: ${filePath}`)
    return true
  }

  return false
}

// Main execution
console.log("🔍 Searching for files with incorrect Image imports...")

const projectRoot = process.cwd()
const filesWithIncorrectImports = findFilesWithIncorrectImports(projectRoot)

if (filesWithIncorrectImports.length === 0) {
  console.log("✅ No files found with incorrect Image imports!")
} else {
  console.log(`❌ Found ${filesWithIncorrectImports.length} files with incorrect imports:`)
  filesWithIncorrectImports.forEach((file) => {
    console.log(`  - ${file}`)
  })

  console.log("\n🔧 Fixing imports...")
  let fixedCount = 0

  filesWithIncorrectImports.forEach((file) => {
    if (fixImageImports(file)) {
      fixedCount++
    }
  })

  console.log(`\n✅ Fixed ${fixedCount} files!`)
}
